<?php

namespace App\Http\Controllers;

use App\Models\BailApplication;
use App\Models\BailCheckin;
use App\Models\BailCondition;
use App\Models\CaseSuspect;
use App\Models\PoliceStation;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class BailCheckinController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = BailCheckin::query()
            ->with(['suspect', 'bailApplication', 'policeStation', 'recordedBy']);
            
        // Filter by police station if provided
        if ($request->has('police_station_id')) {
            $query->where('police_station_id', $request->police_station_id);
        }
        
        // Filter by date range
        if ($request->has('date_from')) {
            $query->whereDate('scheduled_time', '>=', $request->date_from);
        }
        
        if ($request->has('date_to')) {
            $query->whereDate('scheduled_time', '<=', $request->date_to);
        }
        
        // Filter by status
        if ($request->has('status') && $request->status != 'all') {
            $query->where('status', $request->status);
        }
        
        // Filter by suspect
        if ($request->has('suspect_id')) {
            $query->where('suspect_id', $request->suspect_id);
        }
        
        // Filter by bail application
        if ($request->has('bail_application_id')) {
            $query->where('bail_application_id', $request->bail_application_id);
        }
        
        // Default sorting
        $query->orderBy('scheduled_time', 'desc');
        
        // Get paginated results
        $checkins = $query->paginate(15)->withQueryString();
        
        // Get police stations for filter dropdown
        $policeStations = PoliceStation::orderBy('name')->get();
        
        // Get suspects with active bail for filter dropdown
        $suspects = CaseSuspect::whereHas('bailApplications', function($query) {
            $query->where('status', 'approved')
                  ->where(function($q) {
                      $q->whereNull('expiry_date')
                        ->orWhere('expiry_date', '>=', Carbon::today());
                  });
        })->get();
        
        return view('bail.checkins.index', compact('checkins', 'policeStations', 'suspects'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        // Get police stations for dropdown
        $policeStations = PoliceStation::orderBy('name')->get();
        
        // Get suspects with active bail conditions
        $suspects = CaseSuspect::whereHas('bailApplications', function($query) {
            $query->where('status', 'approved')
                  ->where(function($q) {
                      $q->whereNull('expiry_date')
                        ->orWhere('expiry_date', '>=', Carbon::today());
                  });
        })->get();
        
        // Pre-select suspect if provided in request
        $selectedSuspect = null;
        $selectedBailApplication = null;
        
        if ($request->has('suspect_id')) {
            $selectedSuspect = CaseSuspect::with(['bailApplications' => function($query) {
                $query->where('status', 'approved')
                      ->where(function($q) {
                          $q->whereNull('expiry_date')
                            ->orWhere('expiry_date', '>=', Carbon::today());
                      });
            }])->find($request->suspect_id);
            
            if ($selectedSuspect && $selectedSuspect->bailApplications->count() > 0) {
                $selectedBailApplication = $selectedSuspect->bailApplications->first();
            }
        }
        
        // Get officers for dropdown
        $officers = User::where('is_officer', true)->orderBy('name')->get();
        
        return view('bail.checkins.create', compact('policeStations', 'suspects', 'selectedSuspect', 'selectedBailApplication', 'officers'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'bail_application_id' => 'required|exists:bail_applications,id',
            'suspect_id' => 'required|exists:case_suspects,id',
            'police_station_id' => 'required|exists:police_stations,id',
            'checkin_time' => 'required|date',
            'scheduled_time' => 'nullable|date',
            'is_present' => 'boolean',
            'status' => 'required|in:scheduled,present,missed,excused',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            DB::beginTransaction();
            
            $checkin = new BailCheckin();
            $checkin->bail_application_id = $request->bail_application_id;
            $checkin->suspect_id = $request->suspect_id;
            $checkin->police_station_id = $request->police_station_id;
            $checkin->checkin_time = $request->checkin_time;
            $checkin->scheduled_time = $request->scheduled_time;
            $checkin->is_present = $request->has('is_present');
            $checkin->status = $request->status;
            $checkin->notes = $request->notes;
            $checkin->recorded_by = Auth::id();
            $checkin->created_by = Auth::id();
            $checkin->updated_by = Auth::id();
            $checkin->save();
            
            // If the suspect is present, update the bail condition's completed check-ins count
            if ($checkin->is_present) {
                $bailCondition = BailCondition::where('bail_application_id', $checkin->bail_application_id)
                    ->where('condition_type', 'reporting')
                    ->where('requires_reporting', true)
                    ->first();
                    
                if ($bailCondition) {
                    $bailCondition->completed_checkins += 1;
                    $bailCondition->save();
                }
            }
            
            DB::commit();
            
            return redirect()->route('bail.checkins.index')
                ->with('success', 'Bail check-in recorded successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Error recording check-in: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $checkin = BailCheckin::with([
            'suspect', 
            'bailApplication', 
            'policeStation', 
            'recordedBy',
            'createdByUser',
            'updatedByUser'
        ])->findOrFail($id);
        
        return view('bail.checkins.show', compact('checkin'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $checkin = BailCheckin::with(['suspect', 'bailApplication', 'policeStation'])->findOrFail($id);
        $policeStations = PoliceStation::orderBy('name')->get();
        $officers = User::where('is_officer', true)->orderBy('name')->get();
        
        return view('bail.checkins.edit', compact('checkin', 'policeStations', 'officers'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $validator = Validator::make($request->all(), [
            'police_station_id' => 'required|exists:police_stations,id',
            'checkin_time' => 'required|date',
            'scheduled_time' => 'nullable|date',
            'is_present' => 'boolean',
            'status' => 'required|in:scheduled,present,missed,excused',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            DB::beginTransaction();
            
            $checkin = BailCheckin::findOrFail($id);
            $oldStatus = $checkin->status;
            $wasPresent = $checkin->is_present;
            
            $checkin->police_station_id = $request->police_station_id;
            $checkin->checkin_time = $request->checkin_time;
            $checkin->scheduled_time = $request->scheduled_time;
            $checkin->is_present = $request->has('is_present');
            $checkin->status = $request->status;
            $checkin->notes = $request->notes;
            $checkin->updated_by = Auth::id();
            $checkin->save();
            
            // Update the bail condition's completed check-ins count if presence status changed
            if ($wasPresent != $checkin->is_present) {
                $bailCondition = BailCondition::where('bail_application_id', $checkin->bail_application_id)
                    ->where('condition_type', 'reporting')
                    ->where('requires_reporting', true)
                    ->first();
                    
                if ($bailCondition) {
                    if ($checkin->is_present && !$wasPresent) {
                        $bailCondition->completed_checkins += 1;
                    } elseif (!$checkin->is_present && $wasPresent) {
                        $bailCondition->completed_checkins = max(0, $bailCondition->completed_checkins - 1);
                    }
                    $bailCondition->save();
                }
            }
            
            DB::commit();
            
            return redirect()->route('bail.checkins.index')
                ->with('success', 'Bail check-in updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Error updating check-in: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            DB::beginTransaction();
            
            $checkin = BailCheckin::findOrFail($id);
            
            // If the check-in was marked as present, decrement the completed check-ins count
            if ($checkin->is_present) {
                $bailCondition = BailCondition::where('bail_application_id', $checkin->bail_application_id)
                    ->where('condition_type', 'reporting')
                    ->where('requires_reporting', true)
                    ->first();
                    
                if ($bailCondition) {
                    $bailCondition->completed_checkins = max(0, $bailCondition->completed_checkins - 1);
                    $bailCondition->save();
                }
            }
            
            $checkin->delete();
            
            DB::commit();
            
            return redirect()->route('bail.checkins.index')
                ->with('success', 'Bail check-in deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Error deleting check-in: ' . $e->getMessage());
        }
    }
    
    /**
     * Display today's check-ins for a specific police station.
     */
    public function todayCheckins(Request $request)
    {
        // Default to the user's police station if they have one assigned
        $policeStationId = $request->police_station_id;
        if (!$policeStationId && Auth::user()->police_station_id) {
            $policeStationId = Auth::user()->police_station_id;
        }
        
        // Get all police stations for the dropdown
        $policeStations = PoliceStation::orderBy('name')->get();
        
        // Get today's scheduled check-ins
        $today = Carbon::today();
        $query = BailCheckin::with(['suspect', 'bailApplication', 'policeStation'])
            ->whereDate('scheduled_time', $today);
            
        // Filter by police station if provided
        if ($policeStationId) {
            $query->where('police_station_id', $policeStationId);
        }
        
        // Get check-ins grouped by status
        $scheduled = clone $query;
        $scheduled = $scheduled->where('status', 'scheduled')->get();
        
        $present = clone $query;
        $present = $present->where('status', 'present')->get();
        
        $missed = clone $query;
        $missed = $missed->where('status', 'missed')->get();
        
        $excused = clone $query;
        $excused = $excused->where('status', 'excused')->get();
        
        // Get suspects who need to check in today based on their bail conditions
        $suspectsDueToday = CaseSuspect::whereHas('bailApplications', function($query) {
            $query->where('status', 'approved')
                  ->where(function($q) {
                      $q->whereNull('expiry_date')
                        ->orWhere('expiry_date', '>=', Carbon::today());
                  });
        })
        ->with(['bailApplications' => function($query) {
            $query->where('status', 'approved')
                  ->where(function($q) {
                      $q->whereNull('expiry_date')
                        ->orWhere('expiry_date', '>=', Carbon::today());
                  });
        }])
        ->get()
        ->filter(function($suspect) {
            // Check if any of the suspect's bail conditions require reporting today
            foreach ($suspect->bailApplications as $application) {
                $condition = BailCondition::where('bail_application_id', $application->id)
                    ->where('condition_type', 'reporting')
                    ->where('requires_reporting', true)
                    ->first();
                    
                if ($condition && $condition->requires_check_in_today) {
                    return true;
                }
            }
            return false;
        });
        
        return view('bail.checkins.today', compact(
            'policeStations', 
            'policeStationId', 
            'scheduled', 
            'present', 
            'missed', 
            'excused',
            'suspectsDueToday'
        ));
    }
    
    /**
     * Generate reporting statistics for management.
     */
    public function reportingStats(Request $request)
    {
        // Set default date range to last 30 days if not provided
        $startDate = $request->date_from ? Carbon::parse($request->date_from) : Carbon::now()->subDays(30);
        $endDate = $request->date_to ? Carbon::parse($request->date_to) : Carbon::now();
        
        // Get all police stations for filtering
        $policeStations = PoliceStation::orderBy('name')->get();
        
        // Build query for check-ins within the date range
        $query = BailCheckin::whereBetween('scheduled_time', [$startDate->startOfDay(), $endDate->endOfDay()]);
        
        // Filter by police station if provided
        if ($request->has('police_station_id')) {
            $query->where('police_station_id', $request->police_station_id);
        }
        
        // Get total counts by status
        $totalScheduled = (clone $query)->where('status', 'scheduled')->count();
        $totalPresent = (clone $query)->where('status', 'present')->count();
        $totalMissed = (clone $query)->where('status', 'missed')->count();
        $totalExcused = (clone $query)->where('status', 'excused')->count();
        $total = $totalScheduled + $totalPresent + $totalMissed + $totalExcused;
        
        // Calculate compliance rate
        $complianceRate = $total > 0 ? round(($totalPresent + $totalExcused) / $total * 100, 2) : 0;
        
        // Get top 5 suspects with most missed check-ins
        $suspectsMostMissed = BailCheckin::whereBetween('scheduled_time', [$startDate->startOfDay(), $endDate->endOfDay()])
            ->where('status', 'missed')
            ->select('suspect_id', DB::raw('count(*) as missed_count'))
            ->groupBy('suspect_id')
            ->orderBy('missed_count', 'desc')
            ->limit(5)
            ->with('suspect')
            ->get();
            
        // Get check-ins by day for chart
        $dailyStats = BailCheckin::whereBetween('scheduled_time', [$startDate->startOfDay(), $endDate->endOfDay()])
            ->select(
                DB::raw('DATE(scheduled_time) as date'),
                DB::raw('SUM(CASE WHEN status = "scheduled" THEN 1 ELSE 0 END) as scheduled_count'),
                DB::raw('SUM(CASE WHEN status = "present" THEN 1 ELSE 0 END) as present_count'),
                DB::raw('SUM(CASE WHEN status = "missed" THEN 1 ELSE 0 END) as missed_count'),
                DB::raw('SUM(CASE WHEN status = "excused" THEN 1 ELSE 0 END) as excused_count')
            )
            ->groupBy('date')
            ->orderBy('date')
            ->get();
            
        // Get check-ins by police station
        $stationStats = BailCheckin::whereBetween('scheduled_time', [$startDate->startOfDay(), $endDate->endOfDay()])
            ->select(
                'police_station_id',
                DB::raw('COUNT(*) as total_count'),
                DB::raw('SUM(CASE WHEN status = "present" THEN 1 ELSE 0 END) as present_count'),
                DB::raw('SUM(CASE WHEN status = "missed" THEN 1 ELSE 0 END) as missed_count')
            )
            ->groupBy('police_station_id')
            ->with('policeStation')
            ->get()
            ->map(function($item) {
                $item->compliance_rate = $item->total_count > 0 ? round($item->present_count / $item->total_count * 100, 2) : 0;
                return $item;
            });
            
        return view('bail.checkins.stats', compact(
            'policeStations',
            'startDate',
            'endDate',
            'totalScheduled',
            'totalPresent',
            'totalMissed',
            'totalExcused',
            'total',
            'complianceRate',
            'suspectsMostMissed',
            'dailyStats',
            'stationStats'
        ));
    }
}
