<?php

namespace App\Http\Controllers;

use App\Models\CaseRecord;
use App\Models\PoliceStation;
use App\Models\User;
use App\Traits\StationDataFilter;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class CaseController extends Controller
{
    use StationDataFilter;

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Start with station-filtered cases
        $query = $this->getFilteredCases()->with(['station', 'reportingOfficer', 'leadInvestigator']);
        
        // Filter by status if provided
        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }
        
        // Filter by station if provided (only allow if user has multi-station access)
        if ($request->has('station_id') && $request->station_id && Auth::user()->hasMultiStationAccess()) {
            $query->where('station_id', $request->station_id);
        }
        
        // Filter by priority if provided
        if ($request->has('priority') && $request->priority) {
            $query->where('priority', $request->priority);
        }
        
        // Filter by crime type if provided
        if ($request->has('crime_type') && $request->crime_type) {
            $query->where('crime_type', $request->crime_type);
        }
        
        // Filter by date range if provided
        if ($request->has('date_from') && $request->date_from) {
            $query->whereDate('reported_date', '>=', $request->date_from);
        }
        
        if ($request->has('date_to') && $request->date_to) {
            $query->whereDate('reported_date', '<=', $request->date_to);
        }
        
        // Filter by search term if provided
        if ($request->has('search') && $request->search) {
            $searchTerm = '%' . $request->search . '%';
            $query->where(function($q) use ($searchTerm) {
                $q->where('case_number', 'like', $searchTerm)
                  ->orWhere('title', 'like', $searchTerm)
                  ->orWhere('description', 'like', $searchTerm)
                  ->orWhere('incident_location', 'like', $searchTerm);
            });
        }
        
        // Default sorting by reported_date desc
        $query->orderBy('reported_date', 'desc');
        
        $cases = $query->paginate(15);
        
        // Get stations based on user access level
        $stations = Auth::user()->hasMultiStationAccess() ? 
            PoliceStation::all() : 
            PoliceStation::where('id', Auth::user()->station_id)->get();
            
        // Get officers from accessible stations only
        $officers = $this->getFilteredOfficers()->get();
        
        // Define crime types, priorities, and statuses
        $crimeTypes = [
            'assault' => 'Assault',
            'theft' => 'Theft',
            'robbery' => 'Robbery',
            'burglary' => 'Burglary',
            'fraud' => 'Fraud',
            'homicide' => 'Homicide',
            'sexual_assault' => 'Sexual Assault',
            'drug_related' => 'Drug Related',
            'cybercrime' => 'Cybercrime',
            'domestic_violence' => 'Domestic Violence',
            'other' => 'Other',
        ];
        
        $priorities = [
            'low' => 'Low',
            'medium' => 'Medium',
            'high' => 'High',
            'critical' => 'Critical',
        ];
        
        $statuses = [
            'open' => 'Open',
            'under_investigation' => 'Under Investigation',
            'pending_review' => 'Pending Review',
            'closed' => 'Closed',
            'reopened' => 'Reopened',
            'archived' => 'Archived',
        ];
        
        return view('cases.index', compact('cases', 'stations', 'officers', 'crimeTypes', 'priorities', 'statuses'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Get stations based on user access level
        $stations = Auth::user()->hasMultiStationAccess() ? 
            PoliceStation::all() : 
            PoliceStation::where('id', Auth::user()->station_id)->get();
            
        // Get officers from accessible stations only
        $officers = $this->getFilteredOfficers()->get();
        
        $crimeTypes = [
            'assault' => 'Assault',
            'theft' => 'Theft',
            'robbery' => 'Robbery',
            'burglary' => 'Burglary',
            'fraud' => 'Fraud',
            'homicide' => 'Homicide',
            'sexual_assault' => 'Sexual Assault',
            'drug_related' => 'Drug Related',
            'cybercrime' => 'Cybercrime',
            'domestic_violence' => 'Domestic Violence',
            'other' => 'Other',
        ];
        
        $priorities = [
            'low' => 'Low',
            'medium' => 'Medium',
            'high' => 'High',
            'critical' => 'Critical',
        ];
        
        $statuses = [
            'open' => 'Open',
            'under_investigation' => 'Under Investigation',
            'pending_review' => 'Pending Review',
            'closed' => 'Closed',
            'reopened' => 'Reopened',
            'archived' => 'Archived',
        ];
        
        return view('cases.create', compact('stations', 'officers', 'crimeTypes', 'priorities', 'statuses'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'station_id' => 'required|exists:police_stations,id',
            'reporting_officer_id' => 'required|exists:users,id',
            'lead_investigator_id' => 'required|exists:users,id',
            'incident_date' => 'required|date',
            'incident_time' => 'nullable|date_format:H:i',
            'incident_location' => 'required|string|max:255',
            'crime_type' => 'required|string|max:50',
            'crime_category' => 'nullable|string|max:50',
            'status' => 'required|string|max:50',
            'priority' => 'required|string|max:20',
            'is_sensitive' => 'boolean',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
        
        DB::beginTransaction();
        
        try {
            // Generate a unique case number
            $caseNumber = 'CASE-' . date('Y') . '-' . Str::padLeft(CaseRecord::count() + 1, 5, '0');
            
            // Create the case record
            $case = new CaseRecord();
            $case->case_number = $caseNumber;
            $case->title = $request->title;
            $case->description = $request->description;
            $case->station_id = $request->station_id;
            $case->reporting_officer_id = $request->reporting_officer_id;
            $case->lead_investigator_id = $request->lead_investigator_id;
            $case->incident_date = $request->incident_date;
            $case->incident_time = $request->incident_time ? $request->incident_date . ' ' . $request->incident_time : null;
            $case->incident_location = $request->incident_location;
            $case->incident_coordinates = $request->incident_coordinates;
            $case->crime_type = $request->crime_type;
            $case->crime_category = $request->crime_category;
            $case->status = $request->status;
            $case->priority = $request->priority;
            $case->reported_date = now();
            $case->is_sensitive = $request->has('is_sensitive');
            $case->save();
            
            DB::commit();
            
            return redirect()->route('cases.show', $case->id)
                ->with('success', 'Case created successfully with case number: ' . $caseNumber);
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Error creating case: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $case = $this->getFilteredCases()->with([
            'station',
            'reportingOfficer',
            'leadInvestigator',
            'updates' => function($query) {
                $query->orderBy('created_at', 'desc');
            },
            'updates.officer',
            'createdByUser',
            'updatedByUser'
        ])->findOrFail($id);
        
        // Check if user can access this case's station data
        if (!Auth::user()->canAccessStationData($case->station_id)) {
            return redirect()->route('cases.index')
                ->with('error', 'You do not have permission to view this case.');
        }
        
        // Count related items
        $case->evidences_count = $case->evidences()->count();
        $case->suspects_count = $case->suspects()->count();
        $case->victims_count = $case->victims()->count();
        $case->statements_count = $case->statements()->count();
        $case->updates_count = $case->updates()->count();
        
        // Define crime types, priorities, and statuses for badges and display
        $crimeTypes = [
            'assault' => 'Assault',
            'theft' => 'Theft',
            'robbery' => 'Robbery',
            'burglary' => 'Burglary',
            'fraud' => 'Fraud',
            'homicide' => 'Homicide',
            'sexual_assault' => 'Sexual Assault',
            'drug_related' => 'Drug Related',
            'cybercrime' => 'Cybercrime',
            'domestic_violence' => 'Domestic Violence',
            'other' => 'Other',
        ];
        
        return view('cases.show', compact('case', 'crimeTypes'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $case = CaseRecord::findOrFail($id);
        
        // Get stations based on user access level
        $stations = Auth::user()->hasMultiStationAccess() ? 
            PoliceStation::all() : 
            PoliceStation::where('id', Auth::user()->station_id)->get();
            
        // Get officers from accessible stations only
        $officers = $this->getFilteredOfficers()->get();
        
        $crimeTypes = [
            'assault' => 'Assault',
            'theft' => 'Theft',
            'robbery' => 'Robbery',
            'burglary' => 'Burglary',
            'fraud' => 'Fraud',
            'homicide' => 'Homicide',
            'sexual_assault' => 'Sexual Assault',
            'drug_related' => 'Drug Related',
            'cybercrime' => 'Cybercrime',
            'domestic_violence' => 'Domestic Violence',
            'other' => 'Other',
        ];
        
        $priorities = [
            'low' => 'Low',
            'medium' => 'Medium',
            'high' => 'High',
            'critical' => 'Critical',
        ];
        
        $statuses = [
            'open' => 'Open',
            'under_investigation' => 'Under Investigation',
            'pending_review' => 'Pending Review',
            'closed' => 'Closed',
            'reopened' => 'Reopened',
            'archived' => 'Archived',
        ];
        
        return view('cases.edit', compact('case', 'stations', 'officers', 'crimeTypes', 'priorities', 'statuses'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'station_id' => 'required|exists:police_stations,id',
            'reporting_officer_id' => 'required|exists:users,id',
            'lead_investigator_id' => 'required|exists:users,id',
            'incident_date' => 'required|date',
            'incident_time' => 'nullable|date_format:H:i',
            'incident_location' => 'required|string|max:255',
            'crime_type' => 'required|string|max:50',
            'crime_category' => 'nullable|string|max:50',
            'status' => 'required|string|max:50',
            'priority' => 'required|string|max:20',
            'is_sensitive' => 'boolean',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
        
        DB::beginTransaction();
        
        try {
            $case = CaseRecord::findOrFail($id);
            $oldStatus = $case->status;
            
            // Update the case record
            $case->title = $request->title;
            $case->description = $request->description;
            $case->station_id = $request->station_id;
            $case->reporting_officer_id = $request->reporting_officer_id;
            $case->lead_investigator_id = $request->lead_investigator_id;
            $case->incident_date = $request->incident_date;
            $case->incident_time = $request->incident_time ? $request->incident_date . ' ' . $request->incident_time : null;
            $case->incident_location = $request->incident_location;
            $case->incident_coordinates = $request->incident_coordinates;
            $case->crime_type = $request->crime_type;
            $case->crime_category = $request->crime_category;
            $case->status = $request->status;
            $case->priority = $request->priority;
            $case->is_sensitive = $request->has('is_sensitive');
            
            // If status changed to closed, record closing information
            if ($oldStatus != 'closed' && $request->status == 'closed') {
                $case->closed_date = now();
                $case->closed_by = Auth::id();
                $case->closing_remarks = $request->closing_remarks;
            }
            
            // If status changed from closed to something else, clear closing information
            if ($oldStatus == 'closed' && $request->status != 'closed') {
                $case->closed_date = null;
                $case->closed_by = null;
                $case->closing_remarks = null;
            }
            
            $case->save();
            
            // Create a case update for the status change if applicable
            if ($oldStatus != $request->status) {
                $case->updates()->create([
                    'update_type' => 'general',
                    'description' => 'Case status changed from ' . $oldStatus . ' to ' . $request->status,
                    'officer_id' => Auth::id(),
                    'update_date' => now(),
                ]);
            }
            
            DB::commit();
            
            return redirect()->route('cases.show', $case->id)
                ->with('success', 'Case updated successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Error updating case: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $case = CaseRecord::findOrFail($id);
            $case->delete();
            
            return redirect()->route('cases.index')
                ->with('success', 'Case deleted successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error deleting case: ' . $e->getMessage());
        }
    }
    
    /**
     * Display cases assigned to the current officer.
     */
    public function assigned()
    {
        $userId = Auth::id();
        
        // Get assigned cases with station filtering
        $cases = $this->getFilteredCases()
            ->where(function($query) use ($userId) {
                $query->where('lead_investigator_id', $userId)
                      ->orWhere('reporting_officer_id', $userId);
            })
            ->with(['station', 'reportingOfficer', 'leadInvestigator'])
            ->orderBy('reported_date', 'desc')
            ->paginate(15);
        
        return view('cases.assigned', compact('cases'));
    }
    
    /**
     * Display sensitive/classified cases.
     */
    public function classified()
    {
        // Check if user has permission to view classified cases
        if (!Auth::user()->can('view-classified-cases')) {
            return redirect()->route('cases.index')
                ->with('error', 'You do not have permission to view classified cases');
        }
        
        // Get classified cases with station filtering
        $cases = $this->getFilteredCases()
            ->where('is_sensitive', true)
            ->with(['station', 'reportingOfficer', 'leadInvestigator'])
            ->orderBy('reported_date', 'desc')
            ->paginate(15);
        
        return view('cases.classified', compact('cases'));
    }
}
