<?php

namespace App\Http\Controllers;

use App\Models\CaseRecord;
use App\Models\CaseEvidence;
use App\Models\EvidenceTransfer;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class CaseEvidenceController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = CaseEvidence::with(['case', 'collector', 'createdByUser', 'updatedByUser']);
        
        // Filter by case if provided
        if ($request->has('case_id') && $request->case_id) {
            $query->where('case_id', $request->case_id);
        }
        
        // Filter by evidence type if provided
        if ($request->has('evidence_type') && $request->evidence_type) {
            $query->where('evidence_type', $request->evidence_type);
        }
        
        // Filter by status if provided
        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }
        
        // Filter by collection date range if provided
        if ($request->has('collection_date_from') && $request->collection_date_from) {
            $query->whereDate('collection_datetime', '>=', $request->collection_date_from);
        }
        
        if ($request->has('collection_date_to') && $request->collection_date_to) {
            $query->whereDate('collection_datetime', '<=', $request->collection_date_to);
        }
        
        // Filter by search term if provided
        if ($request->has('search') && $request->search) {
            $searchTerm = '%' . $request->search . '%';
            $query->where(function($q) use ($searchTerm) {
                $q->where('tag_number', 'like', $searchTerm)
                  ->orWhere('name', 'like', $searchTerm)
                  ->orWhere('description', 'like', $searchTerm)
                  ->orWhere('collection_location', 'like', $searchTerm)
                  ->orWhere('custodian', 'like', $searchTerm);
            });
        }
        
        // Default sorting by collection_datetime desc
        $query->orderBy('collection_datetime', 'desc');
        
        $evidences = $query->paginate(15);
        $cases = CaseRecord::all();
        $officers = User::where('is_officer', true)->get();
        
        $evidenceTypes = [
            'physical' => 'Physical',
            'digital' => 'Digital',
            'documentary' => 'Documentary',
            'biological' => 'Biological',
            'forensic' => 'Forensic',
            'other' => 'Other',
        ];
        
        $statuses = [
            'collected' => 'Collected',
            'in_storage' => 'In Storage',
            'in_analysis' => 'In Analysis',
            'analyzed' => 'Analyzed',
            'in_court' => 'In Court',
            'returned' => 'Returned',
            'destroyed' => 'Destroyed',
        ];
        
        return view('case-evidences.index', compact('evidences', 'cases', 'evidenceTypes', 'statuses', 'officers'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        $caseId = $request->case_id;
        $case = null;
        
        if ($caseId) {
            $case = CaseRecord::findOrFail($caseId);
        }
        
        $cases = CaseRecord::all();
        $officers = User::where('is_officer', true)->get();
        
        $evidenceTypes = [
            'physical' => 'Physical',
            'digital' => 'Digital',
            'documentary' => 'Documentary',
            'biological' => 'Biological',
            'forensic' => 'Forensic',
            'other' => 'Other',
        ];
        
        $statuses = [
            'collected' => 'Collected',
            'in_storage' => 'In Storage',
            'in_analysis' => 'In Analysis',
            'analyzed' => 'Analyzed',
            'in_court' => 'In Court',
            'returned' => 'Returned',
            'destroyed' => 'Destroyed',
        ];
        
        $conditions = [
            'excellent' => 'Excellent',
            'good' => 'Good',
            'fair' => 'Fair',
            'poor' => 'Poor',
            'damaged' => 'Damaged',
            'contaminated' => 'Contaminated',
        ];
        
        return view('case-evidences.create', compact('case', 'cases', 'officers', 'evidenceTypes', 'statuses', 'conditions'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'case_id' => 'required|exists:cases,id',
            'name' => 'required|string|max:255',
            'evidence_type' => 'required|string|max:50',
            'subtype' => 'nullable|string|max:50',
            'description' => 'required|string',
            'collection_datetime' => 'required|date',
            'collection_time' => 'nullable',
            'collection_location' => 'required|string|max:255',
            'collection_details' => 'nullable|string',
            'collected_by_officer_id' => 'required|exists:users,id',
            'storage_location' => 'nullable|string|max:255',
            'status' => 'required|string|max:50',
            'condition' => 'nullable|string|max:50',
            'custodian' => 'nullable|string|max:100',
            'is_sensitive' => 'boolean',
            'photos.*' => 'nullable|file|mimes:jpeg,png,jpg|max:5120',
            'documents.*' => 'nullable|file|mimes:pdf,doc,docx,xls,xlsx,txt|max:10240',
            'notes' => 'nullable|string',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
        
        try {
            $evidence = new CaseEvidence();
            $evidence->case_id = $request->case_id;
            $evidence->tag_number = 'EV-' . date('Ymd') . '-' . Str::random(5);
            $evidence->name = $request->name;
            $evidence->evidence_type = $request->evidence_type;
            $evidence->subtype = $request->subtype;
            $evidence->description = $request->description;
            $evidence->collection_datetime = $request->collection_datetime;
            $evidence->collection_time = $request->collection_time;
            $evidence->collection_location = $request->collection_location;
            $evidence->collection_details = $request->collection_details;
            $evidence->collected_by_officer_id = $request->collected_by_officer_id;
            $evidence->storage_location = $request->storage_location;
            $evidence->status = $request->status;
            $evidence->condition = $request->condition;
            $evidence->custodian = $request->custodian ?? User::find($request->collected_by_officer_id)->name;
            $evidence->is_sensitive = $request->has('is_sensitive');
            $evidence->notes = $request->notes;
            $evidence->created_by_user_id = Auth::id();
            
            // Handle photo uploads
            $photos = [];
            if ($request->hasFile('photos')) {
                foreach ($request->file('photos') as $photo) {
                    if ($photo->isValid()) {
                        $path = $photo->store('evidence/photos', 'public');
                        $photos[] = $path;
                    }
                }
            }
            $evidence->photos = $photos;
            
            // Handle document uploads
            $documents = [];
            if ($request->hasFile('documents')) {
                foreach ($request->file('documents') as $document) {
                    if ($document->isValid()) {
                        $path = $document->store('evidence/documents', 'public');
                        $documents[] = $path;
                    }
                }
            }
            $evidence->documents = $documents;
            
            // Initialize chain of custody
            $chainOfCustody = [
                [
                    'date' => now()->toDateString(),
                    'time' => now()->format('H:i'),
                    'from' => 'Initial Collection',
                    'to' => $evidence->custodian,
                    'reason' => 'collection',
                    'notes' => 'Initial collection of evidence',
                    'recorded_by' => Auth::user()->name,
                    'recorded_at' => now()->toDateTimeString(),
                ]
            ];
            $evidence->chain_of_custody = $chainOfCustody;
            
            $evidence->save();
            
            // Create a case update for the new evidence
            $evidence->case->updates()->create([
                'update_type' => 'evidence',
                'description' => 'New evidence added: ' . $evidence->tag_number . ' - ' . $evidence->name,
                'officer_id' => Auth::id(),
                'update_date' => now(),
            ]);
            
            return redirect()->route('case-evidences.show', $evidence->id)
                ->with('success', 'Evidence added successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error adding evidence: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $evidence = CaseEvidence::with(['case', 'collector', 'createdByUser', 'updatedByUser'])->findOrFail($id);

        // Get all transfers for this evidence
        $transfers = EvidenceTransfer::where('evidence_id', $id)
            ->orderBy('transfer_date', 'desc')
            ->orderBy('transfer_time', 'desc')
            ->get();

        $transferReasons = [
            'analysis' => 'For Analysis',
            'storage' => 'For Storage',
            'court' => 'For Court Proceedings',
            'return' => 'Return to Owner',
            'disposal' => 'For Disposal',
            'other' => 'Other',
        ];

        return view('case-evidences.show', compact('evidence', 'transfers', 'transferReasons'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $evidence = CaseEvidence::findOrFail($id);
        $case = $evidence->case;
        $cases = CaseRecord::all();
        $officers = User::where('is_officer', true)->get();

        $evidenceTypes = [
            'physical' => 'Physical',
            'digital' => 'Digital',
            'documentary' => 'Documentary',
            'biological' => 'Biological',
            'forensic' => 'Forensic',
            'other' => 'Other',
        ];

        $statuses = [
            'collected' => 'Collected',
            'in_storage' => 'In Storage',
            'in_analysis' => 'In Analysis',
            'analyzed' => 'Analyzed',
            'in_court' => 'In Court',
            'returned' => 'Returned',
            'destroyed' => 'Destroyed',
        ];

        $conditions = [
            'excellent' => 'Excellent',
            'good' => 'Good',
            'fair' => 'Fair',
            'poor' => 'Poor',
            'damaged' => 'Damaged',
            'contaminated' => 'Contaminated',
        ];

        return view('case-evidences.edit', compact('evidence', 'case', 'cases', 'officers', 'evidenceTypes', 'statuses', 'conditions'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $validator = Validator::make($request->all(), [
            'case_id' => 'required|exists:cases,id',
            'name' => 'required|string|max:255',
            'evidence_type' => 'required|string|max:50',
            'subtype' => 'nullable|string|max:50',
            'description' => 'required|string',
            'collection_datetime' => 'required|date',
            'collection_time' => 'nullable',
            'collection_location' => 'required|string|max:255',
            'collection_details' => 'nullable|string',
            'collected_by_officer_id' => 'required|exists:users,id',
            'storage_location' => 'nullable|string|max:255',
            'status' => 'required|string|max:50',
            'condition' => 'nullable|string|max:50',
            'custodian' => 'nullable|string|max:100',
            'is_sensitive' => 'boolean',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            $evidence = CaseEvidence::findOrFail($id);
            $evidence->case_id = $request->case_id;
            $evidence->name = $request->name;
            $evidence->evidence_type = $request->evidence_type;
            $evidence->subtype = $request->subtype;
            $evidence->description = $request->description;
            $evidence->collection_datetime = $request->collection_datetime;
            $evidence->collection_time = $request->collection_time;
            $evidence->collection_location = $request->collection_location;
            $evidence->collection_details = $request->collection_details;
            $evidence->collected_by_officer_id = $request->collected_by_officer_id;
            $evidence->storage_location = $request->storage_location;
            $evidence->status = $request->status;
            $evidence->condition = $request->condition;
            $evidence->custodian = $request->custodian;
            $evidence->is_sensitive = $request->has('is_sensitive');
            $evidence->notes = $request->notes;
            $evidence->updated_by_user_id = Auth::id();

            // Handle photo uploads
            if ($request->hasFile('photos')) {
                $photos = $evidence->photos ?? [];
                foreach ($request->file('photos') as $photo) {
                    if ($photo->isValid()) {
                        $path = $photo->store('evidence/photos', 'public');
                        $photos[] = $path;
                    }
                }
                $evidence->photos = $photos;
            }

            // Handle document uploads
            if ($request->hasFile('documents')) {
                $documents = $evidence->documents ?? [];
                foreach ($request->file('documents') as $document) {
                    if ($document->isValid()) {
                        $path = $document->store('evidence/documents', 'public');
                        $documents[] = $path;
                    }
                }
                $evidence->documents = $documents;
            }

            $evidence->save();

            // Create a case update for the evidence update
            $evidence->case->updates()->create([
                'update_type' => 'evidence',
                'description' => 'Evidence #' . $evidence->tag_number . ' updated',
                'officer_id' => Auth::id(),
                'update_date' => now(),
            ]);

            return redirect()->route('case-evidences.show', $evidence->id)
                ->with('success', 'Evidence updated successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error updating evidence: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $evidence = CaseEvidence::findOrFail($id);
            $caseId = $evidence->case_id;
            $tagNumber = $evidence->tag_number;

            // Delete associated files
            $photos = $evidence->photos ?? [];
            foreach ($photos as $photo) {
                Storage::disk('public')->delete($photo);
            }

            $documents = $evidence->documents ?? [];
            foreach ($documents as $document) {
                Storage::disk('public')->delete($document);
            }

            // Delete related transfers
            EvidenceTransfer::where('evidence_id', $id)->delete();

            $evidence->delete();

            // Create a case update for the evidence deletion
            $case = CaseRecord::find($caseId);
            if ($case) {
                $case->updates()->create([
                    'update_type' => 'evidence',
                    'description' => 'Evidence #' . $tagNumber . ' deleted',
                    'officer_id' => Auth::id(),
                    'update_date' => now(),
                ]);
            }

            return redirect()->route('case-evidences.index')
                ->with('success', 'Evidence deleted successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error deleting evidence: ' . $e->getMessage());
        }
    }

    /**
     * Update chain of custody for evidence.
     */
    public function updateCustody(Request $request, string $id)
    {
        $validator = Validator::make($request->all(), [
            'transfer_date' => 'required|date',
            'transfer_time' => 'required|string',
            'from_person' => 'required|string|max:100',
            'to_person' => 'required|string|max:100',
            'reason' => 'required|string|max:50',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            $evidence = CaseEvidence::findOrFail($id);

            // Create a new evidence transfer record
            $transfer = new EvidenceTransfer();
            $transfer->evidence_id = $id;
            $transfer->transfer_date = $request->transfer_date;
            $transfer->transfer_time = $request->transfer_time;
            $transfer->from_person = $request->from_person;
            $transfer->to_person = $request->to_person;
            $transfer->reason = $request->reason;
            $transfer->notes = $request->notes;
            $transfer->recorded_by_user_id = Auth::id();
            $transfer->save();

            // Update the evidence custodian
            $evidence->custodian = $request->to_person;
            $evidence->updated_by_user_id = Auth::id();

            // Add to chain of custody
            $chainOfCustody = $evidence->chain_of_custody ?? [];
            $chainOfCustody[] = [
                'date' => $request->transfer_date,
                'time' => $request->transfer_time,
                'from' => $request->from_person,
                'to' => $request->to_person,
                'reason' => $request->reason,
                'notes' => $request->notes,
                'recorded_by' => Auth::user()->name,
                'recorded_at' => now()->toDateTimeString(),
            ];

            $evidence->chain_of_custody = $chainOfCustody;
            $evidence->save();

            // Create a case update for the custody transfer
            $evidence->case->updates()->create([
                'update_type' => 'evidence',
                'description' => 'Evidence #' . $evidence->tag_number . ' custody transferred from ' . $request->from_person . ' to ' . $request->to_person,
                'officer_id' => Auth::id(),
                'update_date' => now(),
            ]);

            return redirect()->route('case-evidences.show', $evidence->id)
                ->with('success', 'Custody updated successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error updating custody: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove a photo from evidence.
     */
    public function removePhoto(Request $request, string $id)
    {
        try {
            $evidence = CaseEvidence::findOrFail($id);
            $photoIndex = $request->photo_index;

            $photos = $evidence->photos ?? [];
            if (isset($photos[$photoIndex])) {
                $photoPath = $photos[$photoIndex];
                Storage::disk('public')->delete($photoPath);

                array_splice($photos, $photoIndex, 1);
                $evidence->photos = $photos;
                $evidence->updated_by_user_id = Auth::id();
                $evidence->save();

                return redirect()->back()->with('success', 'Photo removed successfully');
            }

            return redirect()->back()->with('error', 'Photo not found');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Error removing photo: ' . $e->getMessage());
        }
    }

    /**
     * Remove an attachment from evidence.
     */
    public function removeDocument(Request $request, string $id)
    {
        try {
            $evidence = CaseEvidence::findOrFail($id);
            $documentIndex = $request->document_index;

            $documents = $evidence->documents ?? [];
            if (isset($documents[$documentIndex])) {
                $documentPath = $documents[$documentIndex];
                Storage::disk('public')->delete($documentPath);

                array_splice($documents, $documentIndex, 1);
                $evidence->documents = $documents;
                $evidence->updated_by_user_id = Auth::id();
                $evidence->save();

                return redirect()->back()->with('success', 'Document removed successfully');
            }

            return redirect()->back()->with('error', 'Document not found');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Error removing document: ' . $e->getMessage());
        }
    }
}
