<?php

namespace App\Http\Controllers;

use App\Models\CaseRecord;
use App\Models\CaseEvidence;
use App\Models\CaseStatement;
use App\Models\CaseVictim;
use App\Models\CaseSuspect;
use App\Models\PoliceStation;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\View;
use Barryvdh\DomPDF\Facade\Pdf as PDF;

class CaseReportController extends Controller
{
    /**
     * Display the report generation form.
     */
    public function index(Request $request)
    {
        // Get all cases for the dropdown
        $cases = CaseRecord::with(['station', 'reportingOfficer', 'leadInvestigator'])
            ->orderBy('case_number', 'desc')
            ->get();
        
        // Get case ID from request if available
        $selectedCaseId = $request->input('case_id');
        
        return view('case-reports.index', compact('cases', 'selectedCaseId'));
    }
    
    /**
     * Generate and display a case report.
     */
    public function generate(Request $request)
    {
        // Validate the request
        $request->validate([
            'case_id' => 'required|exists:cases,id',
            'include_sections' => 'required|array',
            'report_type' => 'required|in:detailed,summary',
            'output_format' => 'required|in:html,pdf',
        ]);
        
        // Get the case with all related data
        $case = CaseRecord::with([
            'station',
            'reportingOfficer',
            'leadInvestigator',
            'statements' => function($query) {
                $query->with('recordingOfficer', 'verifiedByOfficer');
            },
            'evidence' => function($query) {
                $query->with('collectedBy', 'currentCustodian');
            },
            'victims',
            'suspects',
            'updates' => function($query) {
                $query->orderBy('created_at', 'desc');
            }
        ])->findOrFail($request->case_id);
        
        // Define crime types for display
        $crimeTypes = [
            'assault' => 'Assault',
            'theft' => 'Theft',
            'robbery' => 'Robbery',
            'burglary' => 'Burglary',
            'fraud' => 'Fraud',
            'homicide' => 'Homicide',
            'sexual_assault' => 'Sexual Assault',
            'drug_related' => 'Drug Related',
            'cybercrime' => 'Cybercrime',
            'domestic_violence' => 'Domestic Violence',
            'other' => 'Other',
        ];
        
        // Get sections to include
        $includeSections = $request->input('include_sections', []);
        $reportType = $request->input('report_type');
        
        // Create a case update record for the report generation
        $case->updates()->create([
            'update_type' => 'report',
            'description' => 'Case report generated by ' . Auth::user()->name,
            'officer_id' => Auth::id(),
            'update_date' => now(),
        ]);
        
        // Generate the report view
        $view = view('case-reports.report', compact(
            'case', 
            'crimeTypes', 
            'includeSections', 
            'reportType'
        ));
        
        // Return based on requested format
        if ($request->output_format === 'pdf') {
            $pdf = PDF::loadView('case-reports.report', compact(
                'case', 
                'crimeTypes', 
                'includeSections', 
                'reportType'
            ));
            
            return $pdf->download('case_report_' . $case->case_number . '.pdf');
        }
        
        return $view;
    }
}
