<?php

namespace App\Http\Controllers;

use App\Models\CaseRecord;
use App\Models\CaseSuspect;
use App\Models\SuspectStatusUpdate;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class CaseSuspectController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = CaseSuspect::with(['case']);
        
        // Filter by case if provided
        if ($request->has('case_id') && $request->case_id) {
            $query->where('case_id', $request->case_id);
        }
        
        // Filter by arrest status if provided
        if ($request->has('is_arrested')) {
            $query->where('is_arrested', $request->is_arrested == '1');
        }
        
        // Filter by charge status if provided
        if ($request->has('is_charged')) {
            $query->where('is_charged', $request->is_charged == '1');
        }
        
        // Filter by search term if provided
        if ($request->has('search') && $request->search) {
            $searchTerm = '%' . $request->search . '%';
            $query->where(function($q) use ($searchTerm) {
                $q->where('full_name', 'like', $searchTerm)
                  ->orWhere('alias', 'like', $searchTerm)
                  ->orWhere('id_number', 'like', $searchTerm);
            });
        }
        
        // Default sorting by created_at desc
        $query->orderBy('created_at', 'desc');
        
        $suspects = $query->paginate(15);
        $cases = CaseRecord::all();
        
        return view('case-suspects.index', compact('suspects', 'cases'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        $caseId = $request->case_id;
        $case = null;
        
        if ($caseId) {
            $case = CaseRecord::findOrFail($caseId);
        }
        
        $cases = CaseRecord::all();
        $officers = User::where('is_officer', true)->get();
        
        $idTypes = [
            'national_id' => 'National ID',
            'passport' => 'Passport',
            'drivers_license' => 'Driver\'s License',
            'voter_id' => 'Voter ID',
            'other' => 'Other',
        ];
        
        return view('case-suspects.create', compact('case', 'cases', 'officers', 'idTypes'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'case_id' => 'required|exists:cases,id',
            'full_name' => 'required|string|max:200',
            'alias' => 'nullable|string|max:100',
            'date_of_birth' => 'nullable|date',
            'gender' => 'required|string|max:20',
            'address' => 'nullable|string|max:255',
            'phone' => 'nullable|string|max:50',
            'id_type' => 'nullable|string|max:50',
            'id_number' => 'nullable|string|max:50',
            'nationality' => 'nullable|string|max:100',
            'physical_description' => 'nullable|string',
            'is_arrested' => 'boolean',
            'arrest_date' => 'nullable|date',
            'arrested_by_officer_id' => 'nullable|exists:users,id',
            'arrest_location' => 'nullable|string|max:255',
            'is_charged' => 'boolean',
            'charges' => 'nullable|string',
            'is_in_custody' => 'boolean',
            'custody_location' => 'nullable|string|max:255',
            'photo' => 'nullable|file|mimes:jpeg,png,jpg|max:5120',
            'officer_notes' => 'nullable|string',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
        
        try {
            $suspect = new CaseSuspect();
            $suspect->case_id = $request->case_id;
            $suspect->full_name = $request->full_name;
            $suspect->alias = $request->alias;
            $suspect->date_of_birth = $request->date_of_birth;
            $suspect->gender = $request->gender;
            $suspect->address = $request->address;
            $suspect->phone = $request->phone;
            $suspect->id_type = $request->id_type;
            $suspect->id_number = $request->id_number;
            $suspect->nationality = $request->nationality;
            $suspect->physical_description = $request->physical_description;
            $suspect->is_arrested = $request->has('is_arrested');
            $suspect->arrest_date = $request->arrest_date;
            $suspect->arrested_by_officer_id = $request->arrested_by_officer_id;
            $suspect->arrest_location = $request->arrest_location;
            $suspect->is_charged = $request->has('is_charged');
            $suspect->charges = $request->charges;
            $suspect->is_in_custody = $request->has('is_in_custody');
            $suspect->custody_location = $request->custody_location;
            $suspect->officer_notes = $request->officer_notes;
            
            // Handle photo upload
            if ($request->hasFile('photo')) {
                $photoFile = $request->file('photo');
                $photoPath = $photoFile->store('suspects/photos', 'public');
                $suspect->photo_path = $photoPath;
            }
            
            $suspect->save();
            
            // Create a case update for the new suspect
            $suspect->case->updates()->create([
                'update_type' => 'suspect',
                'description' => 'New suspect added: ' . $suspect->full_name,
                'officer_id' => Auth::id(),
                'update_date' => now(),
            ]);
            
            return redirect()->route('case-suspects.show', $suspect->id)
                ->with('success', 'Suspect added successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error adding suspect: ' . $e->getMessage())
                ->withInput();
        }
    }
    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $suspect = CaseSuspect::with(['case', 'arrestedByOfficer'])->findOrFail($id);
        $statusUpdates = $suspect->statusUpdates()->with('officer')->latest()->get();
        return view('case-suspects.show', compact('suspect', 'statusUpdates'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $suspect = CaseSuspect::findOrFail($id);
        $cases = CaseRecord::all();
        $officers = User::where('is_officer', true)->get();
        
        $idTypes = [
            'national_id' => 'National ID',
            'passport' => 'Passport',
            'drivers_license' => 'Driver\'s License',
            'voter_id' => 'Voter ID',
            'other' => 'Other',
        ];
        
        return view('case-suspects.edit', compact('suspect', 'cases', 'officers', 'idTypes'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $validator = Validator::make($request->all(), [
            'case_id' => 'required|exists:cases,id',
            'full_name' => 'required|string|max:200',
            'alias' => 'nullable|string|max:100',
            'date_of_birth' => 'nullable|date',
            'gender' => 'required|string|max:20',
            'address' => 'nullable|string|max:255',
            'phone' => 'nullable|string|max:50',
            'id_type' => 'nullable|string|max:50',
            'id_number' => 'nullable|string|max:50',
            'nationality' => 'nullable|string|max:100',
            'physical_description' => 'nullable|string',
            'is_arrested' => 'boolean',
            'arrest_date' => 'nullable|date',
            'arrested_by_officer_id' => 'nullable|exists:users,id',
            'arrest_location' => 'nullable|string|max:255',
            'is_charged' => 'boolean',
            'charges' => 'nullable|string',
            'is_in_custody' => 'boolean',
            'custody_location' => 'nullable|string|max:255',
            'photo' => 'nullable|file|mimes:jpeg,png,jpg|max:5120',
            'officer_notes' => 'nullable|string',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
        
        try {
            $suspect = CaseSuspect::findOrFail($id);
            
            // Check if arrest status changed
            $arrestStatusChanged = $suspect->is_arrested != $request->has('is_arrested');
            $chargeStatusChanged = $suspect->is_charged != $request->has('is_charged');
            $custodyStatusChanged = $suspect->is_in_custody != $request->has('is_in_custody');
            
            $suspect->case_id = $request->case_id;
            $suspect->full_name = $request->full_name;
            $suspect->alias = $request->alias;
            $suspect->date_of_birth = $request->date_of_birth;
            $suspect->gender = $request->gender;
            $suspect->address = $request->address;
            $suspect->phone = $request->phone;
            $suspect->id_type = $request->id_type;
            $suspect->id_number = $request->id_number;
            $suspect->nationality = $request->nationality;
            $suspect->physical_description = $request->physical_description;
            $suspect->is_arrested = $request->has('is_arrested');
            $suspect->arrest_date = $request->arrest_date;
            $suspect->arrested_by_officer_id = $request->arrested_by_officer_id;
            $suspect->arrest_location = $request->arrest_location;
            $suspect->is_charged = $request->has('is_charged');
            $suspect->charges = $request->charges;
            $suspect->is_in_custody = $request->has('is_in_custody');
            $suspect->custody_location = $request->custody_location;
            $suspect->officer_notes = $request->officer_notes;
            
            // Handle photo upload
            if ($request->hasFile('photo')) {
                // Delete old photo if exists
                if ($suspect->photo_path) {
                    Storage::disk('public')->delete($suspect->photo_path);
                }
                
                $photoFile = $request->file('photo');
                $photoPath = $photoFile->store('suspects/photos', 'public');
                $suspect->photo_path = $photoPath;
            }
            
            $suspect->save();
            
            // Create case updates for significant changes
            if ($arrestStatusChanged) {
                $suspect->case->updates()->create([
                    'update_type' => 'suspect',
                    'description' => $suspect->full_name . ' ' . 
                        ($suspect->is_arrested ? 'has been arrested' : 'arrest status updated'),
                    'officer_id' => Auth::id(),
                    'update_date' => now(),
                ]);
            }
            
            if ($chargeStatusChanged) {
                $suspect->case->updates()->create([
                    'update_type' => 'suspect',
                    'description' => $suspect->full_name . ' ' . 
                        ($suspect->is_charged ? 'has been charged: ' . $suspect->charges : 'charge status updated'),
                    'officer_id' => Auth::id(),
                    'update_date' => now(),
                ]);
            }
            
            if ($custodyStatusChanged) {
                $suspect->case->updates()->create([
                    'update_type' => 'suspect',
                    'description' => $suspect->full_name . ' custody status updated to: ' . 
                        ($suspect->is_in_custody ? 'In Custody' : 'Not In Custody'),
                    'officer_id' => Auth::id(),
                    'update_date' => now(),
                ]);
            }
            
            return redirect()->route('case-suspects.show', $suspect->id)
                ->with('success', 'Suspect updated successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error updating suspect: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $suspect = CaseSuspect::findOrFail($id);
            $caseId = $suspect->case_id;
            $suspectName = $suspect->full_name;
            
            // Delete photo if exists
            if ($suspect->photo_path) {
                Storage::disk('public')->delete($suspect->photo_path);
            }
            
            $suspect->delete();
            
            // Create a case update for the deleted suspect
            CaseRecord::find($caseId)->updates()->create([
                'update_type' => 'suspect',
                'description' => 'Suspect removed: ' . $suspectName,
                'officer_id' => Auth::id(),
                'update_date' => now(),
            ]);
            
            return redirect()->route('cases.show', $caseId)
                ->with('success', 'Suspect deleted successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error deleting suspect: ' . $e->getMessage());
        }
    }
    
    /**
     * Update the suspect's arrest information.
     */
    public function updateArrest(Request $request, string $id)
    {
        $suspect = CaseSuspect::findOrFail($id);
        
        $validator = Validator::make($request->all(), [
            'is_arrested' => 'boolean',
            'arrest_date' => 'required_if:is_arrested,1|nullable|date',
            'arrested_by_officer_id' => 'required_if:is_arrested,1|nullable|exists:users,id',
            'arrest_location' => 'required_if:is_arrested,1|nullable|string|max:255',
            'arrest_notes' => 'nullable|string',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
        
        try {
            // Get the previous status for comparison
            $previousStatus = $suspect->is_arrested;
            
            // Update the suspect's arrest information
            $suspect->is_arrested = $request->has('is_arrested');
            $suspect->arrest_date = $request->arrest_date;
            $suspect->arrested_by_officer_id = $request->arrested_by_officer_id;
            $suspect->arrest_location = $request->arrest_location;
            
            // Append arrest notes to existing notes
            if ($request->arrest_notes) {
                $currentNotes = $suspect->officer_notes ?? '';
                $dateTime = now()->format('Y-m-d H:i');
                $newNotes = $currentNotes . "\n\n[ARREST UPDATE - {$dateTime}]\n" . $request->arrest_notes;
                $suspect->officer_notes = $newNotes;
            }
            
            $suspect->save();
            
            // Create a status update record
            $statusUpdate = new SuspectStatusUpdate([
                'suspect_id' => $suspect->id,
                'officer_id' => Auth::id(),
                'update_type' => 'arrest',
                'new_status' => $suspect->is_arrested,
                'description' => $request->arrest_notes ?? ($suspect->is_arrested ? 'Suspect arrested' : 'Arrest status updated'),
                'location' => $request->arrest_location,
                'update_date' => now(),
            ]);
            $statusUpdate->save();
            
            // Create a case update for the arrest status change
            $suspect->case->updates()->create([
                'update_type' => 'suspect',
                'description' => $suspect->full_name . ' ' . 
                    ($suspect->is_arrested ? 'has been arrested at ' . $suspect->arrest_location : 'arrest status updated'),
                'officer_id' => Auth::id(),
                'update_date' => now(),
            ]);
            
            return redirect()->route('case-suspects.show', $suspect->id)
                ->with('success', 'Arrest information updated successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error updating arrest information: ' . $e->getMessage())
                ->withInput();
        }
    }
    
    /**
     * Update the suspect's charge information.
     */
    public function updateCharge(Request $request, string $id)
    {
        $suspect = CaseSuspect::findOrFail($id);
        
        $validator = Validator::make($request->all(), [
            'is_charged' => 'boolean',
            'charge_date' => 'nullable|date',
            'charges' => 'nullable|string',
            'charge_notes' => 'nullable|string',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
        
        try {
            // Get the previous status for comparison
            $previousStatus = $suspect->is_charged;
            
            // Update the suspect's charge information
            $suspect->is_charged = $request->has('is_charged');
            $suspect->charge_date = $request->charge_date;
            $suspect->charges = $request->charges;
            
            // Append charge notes to existing notes
            if ($request->charge_notes) {
                $currentNotes = $suspect->officer_notes ?? '';
                $dateTime = now()->format('Y-m-d H:i');
                $newNotes = $currentNotes . "\n\n[CHARGE UPDATE - {$dateTime}]\n" . $request->charge_notes;
                $suspect->officer_notes = $newNotes;
            }
            
            $suspect->save();
            
            // Create a status update record
            $statusUpdate = new SuspectStatusUpdate([
                'suspect_id' => $suspect->id,
                'officer_id' => Auth::id(),
                'update_type' => 'charge',
                'new_status' => $suspect->is_charged,
                'description' => $request->charge_notes ?? ($suspect->is_charged ? 'Suspect charged with: ' . $suspect->charges : 'Charge status updated'),
                'update_date' => now(),
            ]);
            $statusUpdate->save();
            
            // Create a case update if the status changed
            if ($previousStatus != $suspect->is_charged) {
                $suspect->case->updates()->create([
                    'update_type' => 'suspect',
                    'description' => $suspect->is_charged ? 
                        'Suspect ' . $suspect->full_name . ' charged with: ' . $suspect->charges : 
                        'Suspect ' . $suspect->full_name . ' charge status updated',
                    'officer_id' => Auth::id(),
                    'update_date' => now(),
                ]);
            }
            
            return redirect()->route('case-suspects.show', $suspect->id)
                ->with('success', 'Charge information updated successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error updating charge information: ' . $e->getMessage())
                ->withInput();
        }
    }
    
    /**
     * Update the suspect's custody information.
     */
    public function updateCustody(Request $request, string $id)
    {
        $suspect = CaseSuspect::findOrFail($id);
        
        $validator = Validator::make($request->all(), [
            'is_in_custody' => 'boolean',
            'custody_location' => 'nullable|string|max:255',
            'custody_notes' => 'nullable|string',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
        
        try {
            // Get the previous status for comparison
            $previousStatus = $suspect->is_in_custody;
            
            // Update the suspect's custody information
            $suspect->is_in_custody = $request->has('is_in_custody');
            $suspect->custody_location = $request->custody_location;
            
            // Append custody notes to existing notes
            if ($request->custody_notes) {
                $currentNotes = $suspect->officer_notes ?? '';
                $dateTime = now()->format('Y-m-d H:i');
                $newNotes = $currentNotes . "\n\n[CUSTODY UPDATE - {$dateTime}]\n" . $request->custody_notes;
                $suspect->officer_notes = $newNotes;
            }
            
            $suspect->save();
            
            // Create a status update record
            $statusUpdate = new SuspectStatusUpdate([
                'suspect_id' => $suspect->id,
                'officer_id' => Auth::id(),
                'update_type' => 'custody',
                'new_status' => $suspect->is_in_custody,
                'description' => $request->custody_notes ?? ($suspect->is_in_custody ? 'Suspect in custody at ' . $suspect->custody_location : 'Suspect not in custody'),
                'location' => $request->custody_location,
                'update_date' => now(),
            ]);
            $statusUpdate->save();
            
            // Create a case update if the status changed
            if ($previousStatus != $suspect->is_in_custody) {
                $suspect->case->updates()->create([
                    'update_type' => 'suspect',
                    'description' => 'Suspect ' . $suspect->full_name . ' custody status updated to: ' . 
                        ($suspect->is_in_custody ? 'In Custody at ' . $suspect->custody_location : 'Not In Custody'),
                    'officer_id' => Auth::id(),
                    'update_date' => now(),
                ]);
            }
            
            return redirect()->route('case-suspects.show', $suspect->id)
                ->with('success', 'Custody information updated successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error updating custody information: ' . $e->getMessage())
                ->withInput();
        }
    }
    
    /**
     * Update the specified resource in storage.
     */
  
    /**
     * Store a newly created resource in storage.
     */
   
}  