<?php

namespace App\Http\Controllers;

use App\Models\CaseRecord;
use App\Models\CaseUpdate;
use App\Models\User;
use App\Traits\StationDataFilter;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class CaseUpdateController extends Controller
{
    use StationDataFilter;

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Get station-filtered case updates based on user's access
        $query = CaseUpdate::with(['case', 'officer'])
            ->whereHas('case', function($q) {
                // Apply station filtering to cases
                $filteredCases = $this->getFilteredCases();
                $q->whereIn('id', $filteredCases->pluck('id'));
            });
        
        // Filter by case if provided (and user has access to it)
        if ($request->has('case_id') && $request->case_id) {
            $accessibleCase = $this->getFilteredCases()->where('id', $request->case_id)->first();
            if ($accessibleCase) {
                $query->where('case_id', $request->case_id);
            }
        }
        
        // Filter by update type if provided
        if ($request->has('update_type') && $request->update_type) {
            $query->where('update_type', $request->update_type);
        }
        
        // Filter by officer if provided (only show officers from accessible stations)
        if ($request->has('officer_id') && $request->officer_id) {
            $accessibleOfficer = $this->getFilteredOfficers()->where('id', $request->officer_id)->first();
            if ($accessibleOfficer) {
                $query->where('officer_id', $request->officer_id);
            }
        }
        
        // Filter by date range if provided
        if ($request->has('start_date') && $request->start_date) {
            $query->whereDate('update_date', '>=', $request->start_date);
        }
        
        if ($request->has('end_date') && $request->end_date) {
            $query->whereDate('update_date', '<=', $request->end_date);
        }
        
        // Filter by search term if provided
        if ($request->has('search') && $request->search) {
            $searchTerm = '%' . $request->search . '%';
            $query->where('description', 'like', $searchTerm);
        }
        
        // Default sorting by update_date desc
        $query->orderBy('update_date', 'desc');
        
        $updates = $query->paginate(20);
        
        // Get only accessible cases and officers for dropdowns
        $cases = $this->getFilteredCases()->get();
        $officers = $this->getFilteredOfficers()->get();
        
        $updateTypes = [
            'general' => 'General Update',
            'statement' => 'Statement',
            'evidence' => 'Evidence',
            'suspect' => 'Suspect',
            'victim' => 'Victim',
            'status' => 'Status Change',
            'court' => 'Court Proceedings',
            'investigation' => 'Investigation',
        ];
        
        return view('case-updates.index', compact('updates', 'cases', 'officers', 'updateTypes'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        $caseId = $request->case_id;
        $case = null;
        
        if ($caseId) {
            // Check if user has access to this case through station filtering
            $case = $this->getFilteredCases()->where('id', $caseId)->first();
            if (!$case) {
                return redirect()->route('case-updates.index')
                    ->with('error', 'You do not have access to this case.');
            }
        }
        
        // Get only accessible cases for dropdown
        $cases = $this->getFilteredCases()->get();
        
        $updateTypes = [
            'general' => 'General Update',
            'statement' => 'Statement',
            'evidence' => 'Evidence',
            'suspect' => 'Suspect',
            'victim' => 'Victim',
            'status' => 'Status Change',
            'court' => 'Court Proceedings',
            'investigation' => 'Investigation',
        ];
        
        return view('case-updates.create', compact('case', 'cases', 'updateTypes'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'case_id' => 'required|exists:cases,id',
            'update_type' => 'required|string|max:50',
            'description' => 'required|string',
            'update_date' => 'required|date',
            'attachment' => 'nullable|file|max:10240',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
        
        // Validate user has access to this case through station filtering
        $accessibleCase = $this->getFilteredCases()->where('id', $request->case_id)->first();
        if (!$accessibleCase) {
            return redirect()->back()
                ->with('error', 'You do not have access to this case.')
                ->withInput();
        }
        
        try {
            $update = new CaseUpdate();
            $update->case_id = $request->case_id;
            $update->update_type = $request->update_type;
            $update->description = $request->description;
            $update->update_date = $request->update_date;
            $update->officer_id = Auth::id();
            
            // Handle attachment upload
            if ($request->hasFile('attachment')) {
                $attachmentFile = $request->file('attachment');
                $attachmentPath = $attachmentFile->store('updates/attachments', 'public');
                $update->attachment_path = $attachmentPath;
            }
            
            $update->save();
            
            return redirect()->route('cases.show', $update->case_id)
                ->with('success', 'Case update added successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error adding case update: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $update = CaseUpdate::with(['case', 'officer'])->findOrFail($id);
        
        // Validate user has access to this case through station filtering
        $accessibleCase = $this->getFilteredCases()->where('id', $update->case_id)->first();
        if (!$accessibleCase) {
            return redirect()->route('case-updates.index')
                ->with('error', 'You do not have access to this case update.');
        }
        
        return view('case-updates.show', compact('update'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $update = CaseUpdate::findOrFail($id);
        
        // Validate user has access to this case through station filtering
        $accessibleCase = $this->getFilteredCases()->where('id', $update->case_id)->first();
        if (!$accessibleCase) {
            return redirect()->route('case-updates.index')
                ->with('error', 'You do not have access to this case update.');
        }
        
        // Check if user can edit this update (own update or has permission)
        if ($update->officer_id != Auth::id() && !Auth::user()->hasRankPermission('edit-all-updates')) {
            return redirect()->route('case-updates.index')
                ->with('error', 'You do not have permission to edit this update.');
        }
        
        // Get only accessible cases for dropdown
        $cases = $this->getFilteredCases()->get();
        
        $updateTypes = [
            'general' => 'General Update',
            'statement' => 'Statement',
            'evidence' => 'Evidence',
            'suspect' => 'Suspect',
            'victim' => 'Victim',
            'status' => 'Status Change',
            'court' => 'Court Proceedings',
            'investigation' => 'Investigation',
        ];
        
        return view('case-updates.edit', compact('update', 'cases', 'updateTypes'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $validator = Validator::make($request->all(), [
            'case_id' => 'required|exists:cases,id',
            'update_type' => 'required|string|max:50',
            'description' => 'required|string',
            'update_date' => 'required|date',
            'attachment' => 'nullable|file|max:10240',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
        
        try {
            $update = CaseUpdate::findOrFail($id);
            
            // Check if user has permission to edit this update
            if ($update->officer_id != Auth::id() && !Auth::user()->can('edit-all-updates')) {
                return redirect()->back()
                    ->with('error', 'You do not have permission to edit this update.');
            }
            
            $update->case_id = $request->case_id;
            $update->update_type = $request->update_type;
            $update->description = $request->description;
            $update->update_date = $request->update_date;
            
            // Handle attachment upload
            if ($request->hasFile('attachment')) {
                // Delete old attachment if exists
                if ($update->attachment_path) {
                    Storage::disk('public')->delete($update->attachment_path);
                }
                
                $attachmentFile = $request->file('attachment');
                $attachmentPath = $attachmentFile->store('updates/attachments', 'public');
                $update->attachment_path = $attachmentPath;
            }
            
            $update->save();
            
            return redirect()->route('case-updates.show', $update->id)
                ->with('success', 'Case update modified successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error updating case update: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $update = CaseUpdate::findOrFail($id);
            $caseId = $update->case_id;
            
            // Validate user has access to this case through station filtering
            $accessibleCase = $this->getFilteredCases()->where('id', $update->case_id)->first();
            if (!$accessibleCase) {
                return redirect()->back()
                    ->with('error', 'You do not have access to this case update.');
            }
            
            // Check if user has permission to delete this update
            if ($update->officer_id != Auth::id() && !Auth::user()->hasRankPermission('delete-all-updates')) {
                return redirect()->back()
                    ->with('error', 'You do not have permission to delete this update.');
            }
            
            // Delete attachment if exists
            if ($update->attachment_path) {
                Storage::disk('public')->delete($update->attachment_path);
            }
            
            $update->delete();
            
            return redirect()->route('cases.show', $caseId)
                ->with('success', 'Case update deleted successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error deleting case update: ' . $e->getMessage());
        }
    }
    
    /**
     * Get updates for a specific case.
     */
    public function getCaseUpdates(Request $request, string $caseId)
    {
        // Validate user has access to this case through station filtering
        $case = $this->getFilteredCases()->where('id', $caseId)->first();
        if (!$case) {
            return redirect()->route('case-updates.index')
                ->with('error', 'You do not have access to this case.');
        }
        
        $query = $case->updates()->with('officer');
        
        // Filter by update type if provided
        if ($request->has('update_type') && $request->update_type) {
            $query->where('update_type', $request->update_type);
        }
        
        // Filter by date range if provided
        if ($request->has('start_date') && $request->start_date) {
            $query->whereDate('update_date', '>=', $request->start_date);
        }
        
        if ($request->has('end_date') && $request->end_date) {
            $query->whereDate('update_date', '<=', $request->end_date);
        }
        
        // Default sorting by update_date desc
        $query->orderBy('update_date', 'desc');
        
        $updates = $query->paginate(20);
        
        $updateTypes = [
            'general' => 'General Update',
            'statement' => 'Statement',
            'evidence' => 'Evidence',
            'suspect' => 'Suspect',
            'victim' => 'Victim',
            'status' => 'Status Change',
            'court' => 'Court Proceedings',
            'investigation' => 'Investigation',
        ];
        
        return view('case-updates.case-updates', compact('updates', 'case', 'updateTypes'));
    }
    
    /**
     * Download update attachment.
     */
    public function downloadAttachment(string $id)
    {
        $update = CaseUpdate::findOrFail($id);
        
        if (!$update->attachment_path) {
            return redirect()->back()->with('error', 'No attachment found for this update.');
        }
        
        // Check if user has access to this case through station filtering
        $accessibleCase = $this->getFilteredCases()->where('id', $update->case_id)->first();
        if (!$accessibleCase) {
            return redirect()->back()->with('error', 'You do not have permission to access this attachment.');
        }
        
        $path = Storage::disk('public')->path($update->attachment_path);
        
        if (!file_exists($path)) {
            return redirect()->back()->with('error', 'Attachment file not found.');
        }
        
        return response()->download($path);
    }
}
