<?php

namespace App\Http\Controllers;

use App\Models\CaseRecord;
use App\Models\CaseVictim;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class CaseVictimController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = CaseVictim::with(['case']);
        
        // Filter by case if provided
        if ($request->has('case_id') && $request->case_id) {
            $query->where('case_id', $request->case_id);
        }
        
        // Filter by victim type if provided
        if ($request->has('victim_type') && $request->victim_type) {
            $query->where('victim_type', $request->victim_type);
        }
        
        // Filter by injury status if provided
        if ($request->has('injury_status') && $request->injury_status) {
            $query->where('injury_status', $request->injury_status);
        }
        
        // Filter by vulnerability status if provided
        if ($request->has('is_vulnerable')) {
            $query->where('is_vulnerable', $request->is_vulnerable == '1');
        }
        
        // Filter by protection status if provided
        if ($request->has('requires_protection')) {
            $query->where('requires_protection', $request->requires_protection == '1');
        }
        
        // Filter by deceased status if provided
        if ($request->has('is_deceased')) {
            $query->where('is_deceased', $request->is_deceased == '1');
        }
        
        // Filter by search term if provided
        if ($request->has('search') && $request->search) {
            $searchTerm = '%' . $request->search . '%';
            $query->where(function($q) use ($searchTerm) {
                $q->where('full_name', 'like', $searchTerm)
                  ->orWhere('organization_name', 'like', $searchTerm)
                  ->orWhere('phone', 'like', $searchTerm)
                  ->orWhere('email', 'like', $searchTerm)
                  ->orWhere('address', 'like', $searchTerm);
            });
        }
        
        // Default sorting by created_at desc
        $query->orderBy('created_at', 'desc');
        
        $victims = $query->paginate(15);
        $cases = CaseRecord::all();
        
        $victimTypes = [
            'individual' => 'Individual',
            'organization' => 'Organization',
        ];
        
        $injuryStatuses = [
            'none' => 'No Injuries',
            'minor' => 'Minor Injuries',
            'moderate' => 'Moderate Injuries',
            'severe' => 'Severe Injuries',
            'critical' => 'Critical Injuries',
            'deceased' => 'Deceased',
        ];
        
        return view('case-victims.index', compact('victims', 'cases', 'victimTypes', 'injuryStatuses'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        $caseId = $request->case_id;
        $case = null;
        
        if ($caseId) {
            $case = CaseRecord::findOrFail($caseId);
        }
        
        $cases = CaseRecord::all();
        $officers = User::where('is_officer', true)->get();
        
        $idTypes = [
            'national_id' => 'National ID',
            'passport' => 'Passport',
            'drivers_license' => 'Driver\'s License',
            'voter_id' => 'Voter ID',
            'other' => 'Other',
        ];
        
        $victimTypes = [
            'individual' => 'Individual',
            'organization' => 'Organization',
        ];
        
        $injuryStatuses = [
            'none' => 'No Injuries',
            'minor' => 'Minor Injuries',
            'moderate' => 'Moderate Injuries',
            'severe' => 'Severe Injuries',
            'critical' => 'Critical Injuries',
            'deceased' => 'Deceased',
        ];
        
        $organizationTypes = [
            'business' => 'Business',
            'government' => 'Government Agency',
            'non_profit' => 'Non-Profit Organization',
            'educational' => 'Educational Institution',
            'healthcare' => 'Healthcare Facility',
            'religious' => 'Religious Institution',
            'other' => 'Other',
        ];
        
        return view('case-victims.create', compact('case', 'cases', 'officers', 'idTypes', 'victimTypes', 
            'injuryStatuses', 'organizationTypes'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $rules = [
            'case_id' => 'required|exists:cases,id',
            'victim_type' => 'required|in:individual,organization',
        ];
        
        // Add rules based on victim type
        if ($request->victim_type == 'individual') {
            $rules = array_merge($rules, [
                'full_name' => 'required|string|max:100',
                'gender' => 'nullable|string|max:20',
                'date_of_birth' => 'nullable|date',
                'age' => 'nullable|integer|min:0|max:120',
                'nationality' => 'nullable|string|max:50',
                'id_type' => 'nullable|string|max:50',
                'id_number' => 'nullable|string|max:50',
                'occupation' => 'nullable|string|max:100',
                'physical_description' => 'nullable|string',
            ]);
        } else {
            $rules = array_merge($rules, [
                'organization_name' => 'required|string|max:100',
                'organization_type' => 'required|string|max:50',
                'registration_number' => 'nullable|string|max:50',
                'contact_person_name' => 'nullable|string|max:100',
                'contact_person_role' => 'nullable|string|max:100',
            ]);
        }
        
        // Common rules for both victim types
        $rules = array_merge($rules, [
            'address' => 'nullable|string|max:255',
            'phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:100',
            'injury_status' => 'required|string|in:none,minor,moderate,severe,critical,deceased',
            'injuries_sustained' => 'nullable|string',
            'medical_facility' => 'nullable|string|max:100',
            'medical_report' => 'nullable|string',
            'relationship_to_suspect' => 'nullable|string',
            'is_vulnerable' => 'boolean',
            'vulnerability_details' => 'nullable|string',
            'requires_protection' => 'boolean',
            'protection_details' => 'nullable|string',
            'incident_date' => 'nullable|date',
            'incident_location' => 'nullable|string|max:255',
            'incident_description' => 'nullable|string',
            'officer_notes' => 'nullable|string',
            'photo' => 'nullable|file|mimes:jpeg,png,jpg|max:5120',
            'evidence_photos.*' => 'nullable|file|mimes:jpeg,png,jpg|max:5120',
        ]);
        
        // Handle deceased victim
        if ($request->injury_status == 'deceased') {
            $rules = array_merge($rules, [
                'is_deceased' => 'boolean',
                'time_of_death' => 'nullable|date',
                'cause_of_death' => 'nullable|string|max:255',
            ]);
        }
        
        $validator = Validator::make($request->all(), $rules);
        
        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
        
        try {
            // Create new victim instance
            $victim = new CaseVictim();
            $victim->case_id = $request->case_id;
            $victim->victim_type = $request->victim_type;
            
            // Set fields based on victim type
            if ($request->victim_type == 'individual') {
                $victim->full_name = $request->full_name;
                $victim->gender = $request->gender;
                $victim->date_of_birth = $request->date_of_birth;
                $victim->age = $request->age;
                $victim->nationality = $request->nationality;
                $victim->id_type = $request->id_type;
                $victim->id_number = $request->id_number;
                $victim->occupation = $request->occupation;
                $victim->physical_description = $request->physical_description;
            } else {
                $victim->organization_name = $request->organization_name;
                $victim->organization_type = $request->organization_type;
                $victim->registration_number = $request->registration_number;
                $victim->contact_person_name = $request->contact_person_name;
                $victim->contact_person_role = $request->contact_person_role;
            }
            
            // Set common fields
            $victim->address = $request->address;
            $victim->phone = $request->phone;
            $victim->email = $request->email;
            $victim->injury_status = $request->injury_status;
            $victim->injuries_sustained = $request->injuries_sustained;
            $victim->medical_facility = $request->medical_facility;
            $victim->relationship_to_suspect = $request->relationship_to_suspect;
            $victim->is_vulnerable = $request->has('is_vulnerable');
            $victim->vulnerability_details = $request->vulnerability_details;
            $victim->vulnerability_type = $request->vulnerability_type;
            $victim->requires_protection = $request->has('requires_protection');
            $victim->protection_details = $request->protection_details;
            $victim->incident_date = $request->incident_date;
            $victim->incident_location = $request->incident_location;
            $victim->incident_description = $request->incident_description;
            $victim->notes = $request->officer_notes;
            $victim->created_by = Auth::id();
            
            // Handle deceased victim
            if ($request->injury_status == 'deceased') {
                $victim->is_deceased = true;
                $victim->time_of_death = $request->time_of_death;
                $victim->cause_of_death = $request->cause_of_death;
            }
            
            // Handle photo upload if provided
            if ($request->hasFile('photo') && $request->file('photo')->isValid()) {
                $path = $request->file('photo')->store('victims/photos', 'public');
                $victim->photo_path = $path;
            }
            
            // Handle evidence photos uploads if provided
            $evidencePhotos = [];
            if ($request->hasFile('evidence_photos')) {
                foreach ($request->file('evidence_photos') as $photo) {
                    if ($photo->isValid()) {
                        $path = $photo->store('victims/evidence_photos', 'public');
                        $evidencePhotos[] = $path;
                    }
                }
                $victim->evidence_photos = $evidencePhotos;
            }
        
            $victim->save();
            
            // Create a case update for the new victim
            $victim->case->updates()->create([
                'update_type' => 'victim',
                'description' => 'New victim added: ' . ($victim->victim_type == 'individual' ? $victim->full_name : $victim->organization_name),
                'officer_id' => Auth::id(),
                'update_date' => now(),
            ]);
            
            // If victim needs protection, create an additional update
            if ($victim->requires_protection) {
                $victim->case->updates()->create([
                    'update_type' => 'victim',
                    'description' => 'Protection required for victim: ' . ($victim->victim_type == 'individual' ? $victim->full_name : $victim->organization_name),
                    'officer_id' => Auth::id(),
                    'update_date' => now(),
                ]);
                if ($victim->needs_protection) {
                    $victim->case->updates()->create([
                        'update_type' => 'victim',
                        'description' => 'Protection required for victim: ' . $victim->full_name,
                        'officer_id' => Auth::id(),
                        'update_date' => now(),
                    ]);
                }
            }
            
            return redirect()->route('case-victims.show', $victim->id)
                ->with('success', 'Victim added successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error adding victim: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $victim = CaseVictim::with(['case', 'protectionOfficer'])->findOrFail($id);
        return view('case-victims.show', compact('victim'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $victim = CaseVictim::findOrFail($id);
        $cases = CaseRecord::all();
        $officers = User::where('is_officer', true)->get();
        
        $idTypes = [
            'national_id' => 'National ID',
            'passport' => 'Passport',
            'drivers_license' => 'Driver\'s License',
            'voter_id' => 'Voter ID',
            'other' => 'Other',
        ];
        
        $victimStatuses = [
            'active' => 'Active',
            'relocated' => 'Relocated',
            'withdrawn' => 'Withdrawn',
            'deceased' => 'Deceased',
            'uncooperative' => 'Uncooperative',
            'protected' => 'Under Protection',
        ];
        
        $vulnerabilityTypes = [
            'minor' => 'Minor (Under 18)',
            'elderly' => 'Elderly',
            'disabled' => 'Person with Disability',
            'mental_health' => 'Mental Health Condition',
            'language_barrier' => 'Language Barrier',
            'foreign_national' => 'Foreign National',
            'other' => 'Other Vulnerability',
        ];
        
        return view('case-victims.edit', compact('victim', 'cases', 'officers', 'idTypes', 'victimStatuses', 'vulnerabilityTypes'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $validator = Validator::make($request->all(), [
            'case_id' => 'required|exists:cases,id',
            'first_name' => 'required|string|max:100',
            'last_name' => 'required|string|max:100',
            'date_of_birth' => 'nullable|date',
            'gender' => 'required|string|max:20',
            'address' => 'nullable|string|max:255',
            'contact_number' => 'nullable|string|max:50',
            'email' => 'nullable|email|max:100',
            'id_type' => 'nullable|string|max:50',
            'id_number' => 'nullable|string|max:50',
            'is_vulnerable' => 'boolean',
            'vulnerability_details' => 'nullable|string',
            'vulnerability_type' => 'nullable|string|max:50',
            'needs_protection' => 'boolean',
            'protection_details' => 'nullable|string',
            'protection_officer_id' => 'nullable|exists:users,id',
            'victim_status' => 'required|string|max:50',
            'photo' => 'nullable|file|mimes:jpeg,png,jpg|max:5120',
            'medical_report' => 'nullable|file|mimes:pdf,doc,docx|max:10240',
            'impact_statement' => 'nullable|file|mimes:pdf,doc,docx|max:10240',
            'notes' => 'nullable|string',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
        
        try {
            $victim = CaseVictim::findOrFail($id);
            
            // Check if protection status changed
            $protectionStatusChanged = $victim->needs_protection != $request->has('needs_protection');
            $vulnerabilityStatusChanged = $victim->is_vulnerable != $request->has('is_vulnerable');
            $victimStatusChanged = $victim->victim_status != $request->victim_status;
            
            $victim->case_id = $request->case_id;
            $victim->first_name = $request->first_name;
            $victim->last_name = $request->last_name;
            $victim->date_of_birth = $request->date_of_birth;
            $victim->gender = $request->gender;
            $victim->address = $request->address;
            $victim->contact_number = $request->contact_number;
            $victim->email = $request->email;
            $victim->id_type = $request->id_type;
            $victim->id_number = $request->id_number;
            $victim->is_vulnerable = $request->has('is_vulnerable');
            $victim->vulnerability_details = $request->vulnerability_details;
            $victim->vulnerability_type = $request->vulnerability_type;
            $victim->needs_protection = $request->has('needs_protection');
            $victim->protection_details = $request->protection_details;
            $victim->protection_officer_id = $request->protection_officer_id;
            $victim->victim_status = $request->victim_status;
            $victim->notes = $request->notes;
            
            // Handle photo upload
            if ($request->hasFile('photo')) {
                // Delete old photo if exists
                if ($victim->photo_path) {
                    Storage::disk('public')->delete($victim->photo_path);
                }
                
                $photoFile = $request->file('photo');
                $photoPath = $photoFile->store('victims/photos', 'public');
                $victim->photo_path = $photoPath;
            }
            
            // Handle medical report upload
            if ($request->hasFile('medical_report')) {
                // Delete old report if exists
                if ($victim->medical_report_path) {
                    Storage::disk('public')->delete($victim->medical_report_path);
                }
                
                $reportFile = $request->file('medical_report');
                $reportPath = $reportFile->store('victims/medical_reports', 'public');
                $victim->medical_report_path = $reportPath;
            }
            
            // Handle impact statement upload
            if ($request->hasFile('impact_statement')) {
                // Delete old statement if exists
                if ($victim->impact_statement_path) {
                    Storage::disk('public')->delete($victim->impact_statement_path);
                }
                
                $statementFile = $request->file('impact_statement');
                $statementPath = $statementFile->store('victims/impact_statements', 'public');
                $victim->impact_statement_path = $statementPath;
            }
            
            $victim->save();
            
            // Create case updates for significant changes
            if ($protectionStatusChanged) {
                $victim->case->updates()->create([
                    'update_type' => 'victim',
                    'description' => $victim->full_name . ' ' . 
                        ($victim->needs_protection ? 'now requires protection' : 'protection status updated'),
                    'officer_id' => Auth::id(),
                    'update_date' => now(),
                ]);
            }
            
            if ($vulnerabilityStatusChanged) {
                $victim->case->updates()->create([
                    'update_type' => 'victim',
                    'description' => $victim->full_name . ' ' . 
                        ($victim->is_vulnerable ? 'marked as vulnerable: ' . $victim->vulnerability_type : 'vulnerability status updated'),
                    'officer_id' => Auth::id(),
                    'update_date' => now(),
                ]);
            }
            
            if ($victimStatusChanged) {
                $victim->case->updates()->create([
                    'update_type' => 'victim',
                    'description' => $victim->full_name . ' status updated to: ' . $victim->victim_status,
                    'officer_id' => Auth::id(),
                    'update_date' => now(),
                ]);
            }
            
            return redirect()->route('case-victims.show', $victim->id)
                ->with('success', 'Victim updated successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error updating victim: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $victim = CaseVictim::findOrFail($id);
            $caseId = $victim->case_id;
            $victimName = $victim->full_name;
            
            // Delete files if they exist
            if ($victim->photo_path) {
                Storage::disk('public')->delete($victim->photo_path);
            }
            
            if ($victim->medical_report_path) {
                Storage::disk('public')->delete($victim->medical_report_path);
            }
            
            if ($victim->impact_statement_path) {
                Storage::disk('public')->delete($victim->impact_statement_path);
            }
            
            $victim->delete();
            
            // Create a case update for the deleted victim
            CaseRecord::find($caseId)->updates()->create([
                'update_type' => 'victim',
                'description' => 'Victim removed: ' . $victimName,
                'officer_id' => Auth::id(),
                'update_date' => now(),
            ]);
            
            return redirect()->route('cases.show', $caseId)
                ->with('success', 'Victim deleted successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error deleting victim: ' . $e->getMessage());
        }
    }
    
    /**
     * Update protection information for a victim.
     */
    public function updateProtection(Request $request, string $id)
    {
        $validator = Validator::make($request->all(), [
            'needs_protection' => 'required|boolean',
            'protection_details' => 'required_if:needs_protection,1|nullable|string',
            'protection_officer_id' => 'required_if:needs_protection,1|nullable|exists:users,id',
            'protection_notes' => 'nullable|string',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
        
        try {
            $victim = CaseVictim::findOrFail($id);
            $victim->needs_protection = $request->needs_protection;
            $victim->protection_details = $request->protection_details;
            $victim->protection_officer_id = $request->protection_officer_id;
            
            // Append protection notes to existing notes
            if ($request->protection_notes) {
                $currentNotes = $victim->notes ?? '';
                $dateTime = now()->format('Y-m-d H:i');
                $newNotes = $currentNotes . "\n\n[PROTECTION UPDATE - {$dateTime}]\n" . $request->protection_notes;
                $victim->notes = $newNotes;
            }
            
            $victim->save();
            
            // Create a case update for the protection status change
            $victim->case->updates()->create([
                'update_type' => 'victim',
                'description' => $victim->full_name . ' ' . 
                    ($victim->needs_protection ? 'protection status updated with officer assigned' : 'protection status removed'),
                'officer_id' => Auth::id(),
                'update_date' => now(),
            ]);
            
            return redirect()->route('case-victims.show', $victim->id)
                ->with('success', 'Protection information updated successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error updating protection information: ' . $e->getMessage())
                ->withInput();
        }
    }
}
