<?php

namespace App\Http\Controllers;

use App\Models\CaseRecord;
use App\Models\CaseEvidence;
use App\Models\CaseStatement;
use App\Models\CaseVictim;
use App\Models\CaseSuspect;
use App\Models\User;
use App\Models\PoliceStation;
use App\Models\BailApplication;
use App\Models\BailCheckin;
use App\Traits\StationDataFilter;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    use StationDataFilter;

    /**
     * Show the application dashboard.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function index()
    {
        $user = Auth::user();
        $rankLevel = $user->getRankLevel();
        $isAdmin = !$user->is_officer;
        $hasMultiStationAccess = $user->hasMultiStationAccess();
        
        // Initialize dashboard data structure
        $dashboardData = [
            'user_info' => [
                'name' => $user->name,
                'rank' => $user->rank,
                'badge_number' => $user->badge_number,
                'station' => $user->station ? $user->station->name : 'Not Assigned',
                'rank_level' => $rankLevel,
                'is_admin' => $isAdmin,
                'has_multi_station_access' => $hasMultiStationAccess,
            ],
            'permissions' => [
                'can_view_all_stations' => $hasMultiStationAccess || $isAdmin,
                'can_manage_cases' => $user->hasRankPermission('case-create') || $isAdmin,
                'can_manage_evidence' => $user->hasRankPermission('evidence-manage') || $isAdmin,
                'can_manage_officers' => $user->hasRankPermission('officer-create') || $isAdmin,
                'can_approve_bail' => $user->hasRankPermission('bail-approve') || $isAdmin,
                'can_view_reports' => $user->hasRankPermission('reports-view') || $isAdmin,
                'can_manage_stations' => $user->hasRankPermission('station-manage') || $isAdmin,
            ]
        ];
        
        // Core Statistics - Always visible but filtered by permissions
        $stats = $this->getCoreStatistics($user);
        
        // Rank-specific dashboard sections
        $sections = $this->getRankSpecificSections($user, $rankLevel);
        
        // Recent activities based on rank and permissions
        $recentActivities = $this->getRecentActivities($user, $rankLevel);
        
        // Performance metrics based on rank
        $performanceMetrics = $this->getPerformanceMetrics($user, $rankLevel);
        
        // Alerts and notifications based on rank
        $alerts = $this->getIntelligentAlerts($user, $rankLevel);
        
        // Quick actions based on rank and permissions
        $quickActions = $this->getQuickActions($user, $rankLevel);
        
        return view('dashboard', compact(
            'dashboardData', 
            'stats', 
            'sections', 
            'recentActivities', 
            'performanceMetrics', 
            'alerts',
            'quickActions'
        ));
    }
    
    /**
     * Get core statistics filtered by user's permissions
     */
    private function getCoreStatistics($user)
    {
        $stats = [];
        $rankLevel = $user->getRankLevel();
        $isAdmin = !$user->is_officer;
        
        // Station statistics - visible to Inspector+ or multi-station users
        if ($rankLevel >= 4 || $user->hasMultiStationAccess() || $isAdmin) {
            $stats['stations'] = [
                'total' => $user->hasMultiStationAccess() || $isAdmin ? 
                    PoliceStation::count() : 
                    ($user->station_id ? 1 : 0),
                'label' => 'Police Stations',
                'icon' => 'fas fa-building',
                'color' => 'info'
            ];
        }
        
        // Officer statistics - visible to Chief Inspector+ 
        if ($rankLevel >= 5 || $isAdmin) {
            $stats['officers'] = [
                'total' => $this->getFilteredOfficers()->count(),
                'label' => 'Officers',
                'icon' => 'fas fa-users',
                'color' => 'primary'
            ];
        }
        
        // Case statistics - visible to all officers
        if ($user->hasRankPermission('case-view') || $isAdmin) {
            $filteredCases = $this->getFilteredCases();
            $stats['cases'] = [
                'total' => $filteredCases->count(),
                'active' => $filteredCases->where('status', 'active')->count(),
                'closed' => $filteredCases->where('status', 'closed')->count(),
                'pending' => $filteredCases->where('status', 'pending')->count(),
                'label' => 'Cases',
                'icon' => 'fas fa-folder-open',
                'color' => 'success'
            ];
        }
        
        // Evidence statistics - visible to Sergeant+
        if ($rankLevel >= 3 || $isAdmin) {
            $stats['evidence'] = [
                'total' => $this->getFilteredEvidence()->count(),
                'in_custody' => $this->getFilteredEvidence()->where('status', 'in_custody')->count(),
                'in_analysis' => $this->getFilteredEvidence()->where('status', 'in_analysis')->count(),
                'label' => 'Evidence Items',
                'icon' => 'fas fa-box',
                'color' => 'warning'
            ];
        }
        
        // Bail statistics - visible to Inspector+
        if ($rankLevel >= 4 || $isAdmin) {
            $filteredBail = $this->getFilteredBailApplications();
            $stats['bail'] = [
                'pending' => $filteredBail->where('status', 'pending')->count(),
                'approved' => $filteredBail->where('status', 'approved')->count(),
                'rejected' => $filteredBail->where('status', 'rejected')->count(),
                'today_checkins' => $filteredBail->whereHas('checkins', function($query) {
                    $query->whereDate('checkin_time', now()->toDateString());
                })->count(),
                'label' => 'Bail Applications',
                'icon' => 'fas fa-gavel',
                'color' => 'danger'
            ];
        }
        
        return $stats;
    }
    
    /**
     * Get rank-specific dashboard sections
     */
    private function getRankSpecificSections($user, $rankLevel)
    {
        $sections = [];
        $isAdmin = !$user->is_officer;
        
        // Constable (1) - Basic case viewing
        if ($rankLevel >= 1 || $isAdmin) {
            $sections['my_cases'] = [
                'title' => 'My Assigned Cases',
                'data' => $this->getFilteredCases()
                    ->where(function($query) use ($user) {
                        $query->where('reporting_officer_id', $user->id)
                              ->orWhere('lead_investigator_id', $user->id);
                    })
                    ->with(['station', 'reportingOfficer'])
                    ->orderBy('created_at', 'desc')
                    ->take(5)
                    ->get(),
                'show' => true
            ];
        }
        
        // Corporal (2) - Case creation and suspect/victim management
        if ($rankLevel >= 2 || $isAdmin) {
            $sections['recent_suspects'] = [
                'title' => 'Recent Suspects',
                'data' => $this->getFilteredCases()
                    ->with(['suspects' => function($query) {
                        $query->orderBy('created_at', 'desc')->take(3);
                    }])
                    ->take(5)
                    ->get()
                    ->pluck('suspects')
                    ->flatten()
                    ->take(10),
                'show' => true
            ];
        }
        
        // Sergeant (3) - Evidence management and reports
        if ($rankLevel >= 3 || $isAdmin) {
            $sections['evidence_status'] = [
                'title' => 'Evidence Status Overview',
                'data' => $this->getFilteredEvidence()
                    ->select('status', DB::raw('count(*) as count'))
                    ->groupBy('status')
                    ->get(),
                'show' => true
            ];
            
            $sections['pending_analysis'] = [
                'title' => 'Evidence Pending Analysis',
                'data' => $this->getFilteredEvidence()
                    ->where('status', 'in_analysis')
                    ->with(['case'])
                    ->orderBy('created_at', 'asc')
                    ->take(5)
                    ->get(),
                'show' => true
            ];
        }
        
        // Inspector (4) - Station management and bail
        if ($rankLevel >= 4 || $isAdmin) {
            $sections['station_overview'] = [
                'title' => 'Station Performance',
                'data' => [
                    'cases_this_month' => $this->getFilteredCases()
                        ->whereMonth('created_at', now()->month)
                        ->count(),
                    'solved_this_month' => $this->getFilteredCases()
                        ->where('status', 'closed')
                        ->whereMonth('updated_at', now()->month)
                        ->count(),
                    'clearance_rate' => $this->calculateClearanceRate(),
                ],
                'show' => true
            ];
            
            $sections['bail_management'] = [
                'title' => 'Bail Applications Requiring Attention',
                'data' => $this->getFilteredBailApplications()
                    ->where('status', 'pending')
                    ->with(['case', 'suspect'])
                    ->orderBy('created_at', 'asc')
                    ->take(5)
                    ->get(),
                'show' => true
            ];
        }
        
        // Chief Inspector (5) - Officer management
        if ($rankLevel >= 5 || $isAdmin) {
            $sections['officer_performance'] = [
                'title' => 'Top Performing Officers',
                'data' => $this->getFilteredOfficers()
                    ->withCount([
                        'leadInvestigatorCases as active_cases' => function($query) use ($user) {
                            if (!$user->hasMultiStationAccess()) {
                                $query->where('station_id', $user->station_id);
                            }
                            $query->whereMonth('created_at', now()->month);
                        }
                    ])
                    ->with('station')
                    ->orderBy('active_cases', 'desc') // Fix: Changed 'active_cases_count' to 'active_cases'
                    ->take(5)
                    ->get(),
                'show' => true
            ];
        }
        
        // Superintendent (6) - Multi-station oversight
        if ($rankLevel >= 6 || $isAdmin) {
            $sections['multi_station_overview'] = [
                'title' => 'Multi-Station Overview',
                'data' => PoliceStation::withCount([
                    'cases' => function($query) {
                        $query->whereMonth('created_at', now()->month);
                    },
                    'officers'
                ])
                ->orderBy('cases_count', 'desc')
                ->get(),
                'show' => true
            ];
        }
        
        // Assistant Superintendent (7) - Advanced multi-station management and analytics
        if ($rankLevel >= 7 || $isAdmin) {
            // Station Performance Comparison
            $sections['station_performance_comparison'] = [
                'title' => 'Station Performance Comparison',
                'data' => PoliceStation::withCount([
                    'cases as total_cases' => function($query) {
                        $query->whereMonth('created_at', now()->month);
                    },
                    'cases as solved_cases' => function($query) {
                        $query->where('status', 'closed')
                              ->whereMonth('updated_at', now()->month);
                    },
                    'cases as pending_cases' => function($query) {
                        $query->whereIn('status', ['open', 'under_investigation']);
                    },
                    'officers as active_officers'
                ])
                ->get()
                ->map(function($station) {
                    $station->clearance_rate = $station->total_cases > 0 ? 
                        round(($station->solved_cases / $station->total_cases) * 100, 1) : 0;
                    $station->cases_per_officer = $station->active_officers > 0 ? 
                        round($station->total_cases / $station->active_officers, 1) : 0;
                    return $station;
                }),
                'show' => true
            ];
            
            // Critical Cases Requiring Attention
            $sections['critical_cases'] = [
                'title' => 'Critical Cases Requiring Attention',
                'data' => $this->getFilteredCases()
                    ->where(function($query) {
                        $query->where('priority', 'high')
                              ->orWhere('created_at', '<', now()->subDays(30))
                              ->orWhere('crime_type', 'murder')
                              ->orWhere('crime_type', 'armed_robbery');
                    })
                    ->whereIn('status', ['open', 'under_investigation'])
                    ->with(['station', 'reportingOfficer', 'leadInvestigator'])
                    ->orderBy('priority', 'desc')
                    ->orderBy('created_at', 'asc')
                    ->take(8)
                    ->get(),
                'show' => true
            ];
            
            // Resource Allocation Analysis
            $sections['resource_allocation'] = [
                'title' => 'Resource Allocation Analysis',
                'data' => [
                    'officer_workload' => $this->getFilteredOfficers()
                        ->withCount([
                            'leadInvestigatorCases as active_cases' => function($query) {
                                $query->whereIn('status', ['open', 'under_investigation']);
                            }
                        ])
                        ->with('station')
                        ->orderBy('active_cases', 'desc') // Fix: Changed 'active_cases_count' to 'active_cases'
                        ->take(10)
                        ->get(),
                    'evidence_backlog' => $this->getFilteredEvidence()
                        ->where('status', 'pending_analysis')
                        ->with(['case.station'])
                        ->orderBy('created_at', 'asc')
                        ->take(10)
                        ->get(),
                    'bail_pending_approval' => $this->getFilteredBailApplications()
                        ->where('status', 'pending')
                        ->with(['case.station', 'suspect'])
                        ->orderBy('created_at', 'asc')
                        ->take(10)
                        ->get()
                ],
                'show' => true
            ];
            
            // Monthly Performance Trends
            $sections['performance_trends'] = [
                'title' => 'Monthly Performance Trends',
                'data' => [
                    'case_resolution_trend' => $this->getCaseResolutionTrend(),
                    'crime_type_distribution' => $this->getCrimeTypeDistribution(),
                    'station_efficiency_ranking' => $this->getStationEfficiencyRanking()
                ],
                'show' => true
            ];
            
            // Operational Alerts
            $sections['operational_alerts'] = [
                'title' => 'Operational Alerts & Recommendations',
                'data' => $this->getOperationalAlerts(),
                'show' => true
            ];
        }
        
        // Commissioner (9) - System-wide analytics
        if ($rankLevel >= 9 || $isAdmin) {
            $sections['system_analytics'] = [
                'title' => 'System-Wide Analytics',
                'data' => [
                    'crime_trends' => $this->getCrimeTrends(),
                    'resource_utilization' => $this->getResourceUtilization(),
                    'performance_metrics' => $this->getSystemPerformanceMetrics(),
                ],
                'show' => true
            ];
        }
        
        return $sections;
    }
    
    /**
     * Get recent activities based on rank
     */
    private function getRecentActivities($user, $rankLevel)
    {
        $activities = [];
        $isAdmin = !$user->is_officer;
        
        // Personal activities for all ranks
        if ($rankLevel >= 1 || $isAdmin) {
            // Recent cases assigned to user
            $recentCases = $this->getFilteredCases()
                ->where(function($query) use ($user) {
                    $query->where('reporting_officer_id', $user->id)
                          ->orWhere('lead_investigator_id', $user->id);
                })
                ->orderBy('updated_at', 'desc')
                ->take(3)
                ->get();
            
            foreach ($recentCases as $case) {
                $activities[] = [
                    'type' => 'case_update',
                    'title' => "Case #{$case->case_number} updated",
                    'description' => $case->title,
                    'time' => $case->updated_at->diffForHumans(),
                    'icon' => 'fas fa-folder-open',
                    'color' => 'primary'
                ];
            }
        }
        
        // Station-wide activities for Inspector+
        if ($rankLevel >= 4 || $isAdmin) {
            $recentStationCases = $this->getFilteredCases()
                ->orderBy('created_at', 'desc')
                ->take(5)
                ->get();
            
            foreach ($recentStationCases as $case) {
                $activities[] = [
                    'type' => 'new_case',
                    'title' => "New case reported",
                    'description' => "Case #{$case->case_number}: {$case->title}",
                    'time' => $case->created_at->diffForHumans(),
                    'icon' => 'fas fa-plus-circle',
                    'color' => 'success'
                ];
            }
        }
        
        // Sort activities by time
        usort($activities, function($a, $b) {
            return strtotime($b['time']) - strtotime($a['time']);
        });
        
        return array_slice($activities, 0, 10);
    }
    
    /**
     * Get performance metrics based on rank
     */
    private function getPerformanceMetrics($user, $rankLevel)
    {
        $metrics = [];
        $isAdmin = !$user->is_officer;
        
        if ($rankLevel >= 3 || $isAdmin) {
            $metrics['case_closure_rate'] = $this->calculateCaseClosureRate($user);
            $metrics['average_case_duration'] = $this->calculateAverageCaseDuration($user);
        }
        
        if ($rankLevel >= 4 || $isAdmin) {
            $metrics['station_efficiency'] = $this->calculateStationEfficiency($user);
            $metrics['resource_utilization'] = $this->calculateResourceUtilization($user);
        }
        
        if ($rankLevel >= 6 || $isAdmin) {
            $metrics['multi_station_comparison'] = $this->getMultiStationComparison();
        }
        
        return $metrics;
    }
    
    /**
     * Get intelligent alerts based on rank and responsibilities
     */
    private function getIntelligentAlerts($user, $rankLevel)
    {
        $alerts = [];
        $isAdmin = !$user->is_officer;
        
        // Overdue cases - for all officers
        if ($rankLevel >= 1 || $isAdmin) {
            $overdueCases = $this->getFilteredCases()
                ->where('status', 'active')
                ->where('created_at', '<', now()->subDays(30))
                ->count();
            
            if ($overdueCases > 0) {
                $alerts[] = [
                    'type' => 'warning',
                    'title' => 'Overdue Cases',
                    'message' => "{$overdueCases} cases are overdue (>30 days)",
                    'action_url' => route('cases.index', ['filter' => 'overdue']),
                    'priority' => 'high'
                ];
            }
        }
        
        // Evidence requiring attention - for Sergeant+
        if ($rankLevel >= 3 || $isAdmin) {
            $pendingEvidence = $this->getFilteredEvidence()
                ->where('status', 'in_analysis')
                ->where('created_at', '<', now()->subDays(7))
                ->count();
            
            if ($pendingEvidence > 0) {
                $alerts[] = [
                    'type' => 'info',
                    'title' => 'Evidence Analysis Pending',
                    'message' => "{$pendingEvidence} evidence items pending analysis >7 days",
                    'action_url' => route('evidence.index', ['status' => 'in_analysis']),
                    'priority' => 'medium'
                ];
            }
        }
        
        // Bail applications requiring approval - for Inspector+
        if ($rankLevel >= 4 || $isAdmin) {
            $pendingBail = $this->getFilteredBailApplications()
                ->where('status', 'pending')
                ->where('created_at', '<', now()->subDays(2))
                ->count();
            
            if ($pendingBail > 0) {
                $alerts[] = [
                    'type' => 'danger',
                    'title' => 'Urgent Bail Applications',
                    'message' => "{$pendingBail} bail applications pending >2 days",
                    'action_url' => route('bail.index', ['status' => 'pending']),
                    'priority' => 'urgent'
                ];
            }
        }
        
        // Station performance alerts - for Superintendent+
        if ($rankLevel >= 6 || $isAdmin) {
            $lowPerformanceStations = $this->getUnderperformingStations();
            if (count($lowPerformanceStations) > 0) {
                $alerts[] = [
                    'type' => 'warning',
                    'title' => 'Station Performance',
                    'message' => count($lowPerformanceStations) . " stations below performance threshold",
                    'action_url' => route('reports.stations'),
                    'priority' => 'medium'
                ];
            }
        }
        
        // Operational Alerts
        $overloadedOfficers = $this->getFilteredOfficers()
            ->withCount([
                'leadInvestigatorCases as active_cases' => function($query) {
                    $query->whereIn('status', ['open', 'under_investigation']);
                }
            ])
            ->having('active_cases', '>', 15)
            ->with('station')
            ->get();

        foreach ($overloadedOfficers as $officer) {
            $alerts[] = [
                'type' => 'warning',
                'icon' => 'exclamation-triangle',
                'title' => 'Overloaded Officer',
                'message' => "Officer {$officer->name} has {$officer->active_cases} active cases",
                'action_url' => route('officers.show', $officer->id),
                'priority' => 'medium'
            ];
        }

        $stalledCases = $this->getFilteredCases()
            ->where('status', 'under_investigation')
            ->where('updated_at', '<', now()->subDays(14))
            ->with(['station', 'leadInvestigator'])
            ->take(10)
            ->get();

        foreach ($stalledCases as $case) {
            $alerts[] = [
                'type' => 'info',
                'icon' => 'clock',
                'title' => 'Stalled Case',
                'message' => "Case #{$case->case_number} has been under investigation for >14 days",
                'action_url' => route('cases.show', $case->id),
                'priority' => 'low'
            ];
        }

        $evidenceDelays = $this->getFilteredEvidence()
            ->where('status', 'pending_analysis')
            ->where('created_at', '<', now()->subDays(7))
            ->with(['case.station'])
            ->take(10)
            ->get();

        foreach ($evidenceDelays as $evidence) {
            $alerts[] = [
                'type' => 'danger',
                'icon' => 'exclamation-circle',
                'title' => 'Evidence Delay',
                'message' => "Evidence #{$evidence->id} has been pending analysis for >7 days",
                'action_url' => route('evidence.show', $evidence->id),
                'priority' => 'high'
            ];
        }

        return $alerts;
    }
    
    /**
     * Get quick actions based on rank and permissions
     */
    private function getQuickActions($user, $rankLevel)
    {
        $actions = [];
        $isAdmin = !$user->is_officer;
        
        // Basic actions for all officers
        if ($rankLevel >= 1 || $isAdmin) {
            $actions[] = [
                'title' => 'View My Cases',
                'url' => route('cases.index', ['assigned_to_me' => true]),
                'icon' => 'fas fa-folder-open',
                'color' => 'primary'
            ];
        }
        
        // Case creation for Corporal+
        if ($rankLevel >= 2 || $isAdmin) {
            $actions[] = [
                'title' => 'Create New Case',
                'url' => route('cases.create'),
                'icon' => 'fas fa-plus',
                'color' => 'success'
            ];
        }
        
        // Evidence management for Sergeant+
        if ($rankLevel >= 3 || $isAdmin) {
            $actions[] = [
                'title' => 'Manage Evidence',
                'url' => route('evidence.index'),
                'icon' => 'fas fa-box',
                'color' => 'warning'
            ];
        }
        
        // Bail management for Inspector+
        if ($rankLevel >= 4 || $isAdmin) {
            $actions[] = [
                'title' => 'Review Bail Applications',
                'url' => route('bail.index', ['status' => 'pending']),
                'icon' => 'fas fa-gavel',
                'color' => 'danger'
            ];
        }
        
        // Officer management for Chief Inspector+
        if ($rankLevel >= 5 || $isAdmin) {
            $actions[] = [
                'title' => 'Manage Officers',
                'url' => route('officers.index'),
                'icon' => 'fas fa-users',
                'color' => 'info'
            ];
        }
        
        // Reports for Sergeant+
        if ($rankLevel >= 3 || $isAdmin) {
            $actions[] = [
                'title' => 'Generate Reports',
                'url' => route('reports.statistics'),
                'icon' => 'fas fa-chart-bar',
                'color' => 'secondary'
            ];
        }
        
        return $actions;
    }
    
    // Helper methods for calculations
    private function calculateClearanceRate()
    {
        $totalCases = $this->getFilteredCases()->count();
        $closedCases = $this->getFilteredCases()->where('status', 'closed')->count();
        return $totalCases > 0 ? round(($closedCases / $totalCases) * 100, 1) : 0;
    }
    
    private function calculateCaseClosureRate($user)
    {
        $userCases = $this->getFilteredCases()
            ->where(function($query) use ($user) {
                $query->where('reporting_officer_id', $user->id)
                      ->orWhere('lead_investigator_id', $user->id);
            });
        
        $total = $userCases->count();
        $closed = $userCases->where('status', 'closed')->count();
        
        return $total > 0 ? round(($closed / $total) * 100, 1) : 0;
    }
    
    private function calculateAverageCaseDuration($user)
    {
        $closedCases = $this->getFilteredCases()
            ->where('status', 'closed')
            ->where(function($query) use ($user) {
                $query->where('reporting_officer_id', $user->id)
                      ->orWhere('lead_investigator_id', $user->id);
            })
            ->get();
        
        if ($closedCases->count() === 0) return 0;
        
        $totalDays = $closedCases->sum(function($case) {
            return $case->created_at->diffInDays($case->updated_at);
        });
        
        return round($totalDays / $closedCases->count(), 1);
    }
    
    private function calculateStationEfficiency($user)
    {
        // Implementation for station efficiency calculation
        return 85.5; // Placeholder
    }
    
    private function calculateResourceUtilization($user)
    {
        // Implementation for resource utilization calculation
        return 78.2; // Placeholder
    }
    
    private function getCaseResolutionTrend()
    {
        $months = [];
        $resolved = [];
        $total = [];
        
        for ($i = 5; $i >= 0; $i--) {
            $date = now()->subMonths($i);
            $monthName = $date->format('M Y');
            
            $totalCases = $this->getFilteredCases()
                ->whereYear('created_at', $date->year)
                ->whereMonth('created_at', $date->month)
                ->count();
                
            $resolvedCases = $this->getFilteredCases()
                ->where('status', 'closed')
                ->whereYear('updated_at', $date->year)
                ->whereMonth('updated_at', $date->month)
                ->count();
            
            $months[] = $monthName;
            $total[] = $totalCases;
            $resolved[] = $resolvedCases;
        }
        
        return [
            'months' => $months,
            'total_cases' => $total,
            'resolved_cases' => $resolved
        ];
    }
    
    private function getCrimeTypeDistribution()
    {
        return $this->getFilteredCases()
            ->select('crime_type', DB::raw('count(*) as count'))
            ->whereMonth('created_at', now()->month)
            ->groupBy('crime_type')
            ->orderBy('count', 'desc')
            ->get()
            ->map(function($item) {
                return [
                    'type' => ucwords(str_replace('_', ' ', $item->crime_type)),
                    'count' => $item->count,
                    'percentage' => 0 // Will be calculated in view
                ];
            });
    }
    
    private function getStationEfficiencyRanking()
    {
        return PoliceStation::withCount([
            'cases as total_cases' => function($query) {
                $query->whereMonth('created_at', now()->month);
            },
            'cases as solved_cases' => function($query) {
                $query->where('status', 'closed')
                      ->whereMonth('updated_at', now()->month);
            },
            'officers as officer_count'
        ])
        ->get()
        ->map(function($station) {
            $station->clearance_rate = $station->total_cases > 0 ? 
                round(($station->solved_cases / $station->total_cases) * 100, 1) : 0;
            $station->efficiency_score = $station->officer_count > 0 ? 
                round(($station->solved_cases / $station->officer_count) * 10, 1) : 0;
            return $station;
        })
        ->sortByDesc('efficiency_score')
        ->values();
    }

    private function getCrimeTrends()
    {
        $trends = [];
        $caseTypes = ['theft', 'assault', 'burglary', 'fraud', 'drug_offense', 'murder', 'armed_robbery'];
        
        foreach ($caseTypes as $type) {
            $monthlyData = [];
            for ($i = 5; $i >= 0; $i--) {
                $date = now()->subMonths($i);
                $count = CaseRecord::where('crime_type', $type)
                    ->whereYear('created_at', $date->year)
                    ->whereMonth('created_at', $date->month)
                    ->count();
                $monthlyData[] = $count;
            }
            $trends[$type] = $monthlyData;
        }
        
        return $trends;
    }

    private function getResourceUtilization()
    {
        return [
            'officer_utilization' => $this->calculateOfficerUtilization(),
            'evidence_processing' => $this->calculateEvidenceProcessingRate(),
            'case_load_distribution' => $this->calculateCaseLoadDistribution()
        ];
    }
    
    private function calculateOfficerUtilization()
    {
        $totalOfficers = User::where('is_officer', true)->count();
        $activeOfficers = User::where('is_officer', true)
            ->whereHas('leadInvestigatorCases', function($query) {
                $query->whereIn('status', ['open', 'under_investigation']);
            })
            ->count();
            
        return $totalOfficers > 0 ? round(($activeOfficers / $totalOfficers) * 100, 1) : 0;
    }
    
    private function calculateEvidenceProcessingRate()
    {
        $totalEvidence = CaseEvidence::count();
        $processedEvidence = CaseEvidence::where('status', 'analyzed')->count();
        
        return $totalEvidence > 0 ? round(($processedEvidence / $totalEvidence) * 100, 1) : 0;
    }
    
    private function calculateCaseLoadDistribution()
    {
        return User::where('is_officer', true)
            ->withCount([
                'leadInvestigatorCases as active_cases' => function($query) {
                    $query->whereIn('status', ['open', 'under_investigation']);
                }
            ])
            ->get()
            ->groupBy(function($officer) {
                if ($officer->active_cases == 0) return 'No Cases';
                if ($officer->active_cases <= 5) return '1-5 Cases';
                if ($officer->active_cases <= 10) return '6-10 Cases';
                if ($officer->active_cases <= 15) return '11-15 Cases';
                return '15+ Cases';
            })
            ->map(function($group) {
                return $group->count();
            });
    }

    private function getSystemPerformanceMetrics()
    {
        return [
            'overall_clearance_rate' => $this->calculateClearanceRate(),
            'average_case_duration' => $this->calculateSystemAverageCaseDuration(),
            'evidence_backlog' => CaseEvidence::where('status', 'pending_analysis')->count(),
            'pending_bail_applications' => BailApplication::where('status', 'pending')->count()
        ];
    }
    
    private function calculateSystemAverageCaseDuration()
    {
        $closedCases = CaseRecord::where('status', 'closed')
            ->whereMonth('updated_at', now()->month)
            ->get();
            
        if ($closedCases->count() === 0) return 0;
        
        $totalDays = $closedCases->sum(function($case) {
            return $case->created_at->diffInDays($case->updated_at);
        });
        
        return round($totalDays / $closedCases->count(), 1);
    }

    private function getUnderperformingStations()
    {
        return PoliceStation::withCount([
            'cases as total_cases' => function($query) {
                $query->whereMonth('created_at', now()->month);
            },
            'cases as solved_cases' => function($query) {
                $query->where('status', 'closed')
                      ->whereMonth('updated_at', now()->month);
            }
        ])
        ->get()
        ->filter(function($station) {
            $clearanceRate = $station->total_cases > 0 ? 
                ($station->solved_cases / $station->total_cases) * 100 : 0;
            return $clearanceRate < 60; // Stations with less than 60% clearance rate
        })
        ->sortBy(function($station) {
            return $station->total_cases > 0 ? 
                ($station->solved_cases / $station->total_cases) * 100 : 0;
        })
        ->values();
    }

    private function getMultiStationComparison()
    {
        return PoliceStation::withCount([
            'cases as monthly_cases' => function($query) {
                $query->whereMonth('created_at', now()->month);
            },
            'cases as solved_cases' => function($query) {
                $query->where('status', 'closed')
                      ->whereMonth('updated_at', now()->month);
            },
            'officers'
        ])
        ->get()
        ->map(function($station) {
            $station->clearance_rate = $station->monthly_cases > 0 ? 
                round(($station->solved_cases / $station->monthly_cases) * 100, 1) : 0;
            return $station;
        });
    }

    private function getOperationalAlerts()
    {
        $alerts = [];

        $overloadedOfficers = $this->getFilteredOfficers()
            ->withCount([
                'leadInvestigatorCases as active_cases' => function($query) {
                    $query->whereIn('status', ['open', 'under_investigation']);
                }
            ])
            ->having('active_cases', '>', 15)
            ->with('station')
            ->get();

        foreach ($overloadedOfficers as $officer) {
            $alerts[] = [
                'type' => 'warning',
                'icon' => 'exclamation-triangle',
                'title' => 'Overloaded Officer',
                'message' => "Officer {$officer->name} has {$officer->active_cases} active cases",
                'action_url' => route('officers.show', $officer->id),
                'priority' => 'medium'
            ];
        }

        $stalledCases = $this->getFilteredCases()
            ->where('status', 'under_investigation')
            ->where('updated_at', '<', now()->subDays(14))
            ->with(['station', 'leadInvestigator'])
            ->take(10)
            ->get();

        foreach ($stalledCases as $case) {
            $alerts[] = [
                'type' => 'info',
                'icon' => 'clock',
                'title' => 'Stalled Case',
                'message' => "Case #{$case->case_number} has been under investigation for >14 days",
                'action_url' => route('cases.show', $case->id),
                'priority' => 'low'
            ];
        }

        $evidenceDelays = $this->getFilteredEvidence()
            ->where('status', 'pending_analysis')
            ->where('created_at', '<', now()->subDays(7))
            ->with(['case.station'])
            ->take(10)
            ->get();

        foreach ($evidenceDelays as $evidence) {
            $alerts[] = [
                'type' => 'danger',
                'icon' => 'exclamation-circle',
                'title' => 'Evidence Delay',
                'message' => "Evidence #{$evidence->id} has been pending analysis for >7 days",
                'action_url' => route('evidence.show', $evidence->id),
                'priority' => 'high'
            ];
        }

        return $alerts;
    }
}
