<?php

namespace App\Http\Controllers;

use App\Models\Evidence;
use App\Models\User;
use App\Traits\StationDataFilter;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class EvidenceController extends Controller
{
    use StationDataFilter;
    
    // Constructor removed - middleware is now defined in routes file
    
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Start with station-filtered evidence
        $query = $this->getFilteredEvidence();
        
        // Apply filters
        if ($request->has('tag_number')) {
            $query->where('tag_number', 'like', '%' . $request->tag_number . '%');
        }
        
        if ($request->has('evidence_type') && $request->evidence_type != '') {
            $query->where('evidence_type', $request->evidence_type);
        }
        
        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }
        
        if ($request->has('case_id') && $request->case_id != '') {
            $query->where('case_id', $request->case_id);
        }
        
        // Get evidence with relationships
        $evidence = $query->with(['collector', 'currentCustodian', 'case'])
            ->orderBy('created_at', 'desc')
            ->paginate(10);
        
        // Get evidence types and statuses for filter dropdowns
        $evidenceTypes = [
            'physical' => 'Physical',
            'digital' => 'Digital',
            'biological' => 'Biological',
            'documentary' => 'Documentary',
            'weapon' => 'Weapon',
            'drug' => 'Drug',
            'other' => 'Other',
        ];
        
        $statuses = [
            'in_custody' => 'In Custody',
            'in_analysis' => 'In Analysis',
            'returned' => 'Returned',
            'destroyed' => 'Destroyed',
            'missing' => 'Missing',
            'transferred' => 'Transferred',
        ];
        
        return view('evidence.index', compact('evidence', 'evidenceTypes', 'statuses'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Get evidence types for dropdown
        $evidenceTypes = [
            'physical' => 'Physical',
            'digital' => 'Digital',
            'biological' => 'Biological',
            'documentary' => 'Documentary',
            'weapon' => 'Weapon',
            'drug' => 'Drug',
            'other' => 'Other',
        ];
        
        // Get statuses for dropdown
        $statuses = [
            'in_custody' => 'In Custody',
            'in_analysis' => 'In Analysis',
            'returned' => 'Returned',
            'destroyed' => 'Destroyed',
            'missing' => 'Missing',
            'transferred' => 'Transferred',
        ];
        
        // Get officers from accessible stations only
        $officers = $this->getFilteredOfficers()->get();
        
        return view('evidence.create', compact('evidenceTypes', 'statuses', 'officers'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Validate the request
        $validated = $request->validate([
            'tag_number' => 'required|string|max:255|unique:evidence,tag_number',
            'evidence_type' => 'required|string',
            'description' => 'required|string',
            'status' => 'required|string',
            'condition' => 'nullable|string|max:255',
            'location' => 'nullable|string|max:255',
            'storage_location' => 'nullable|string|max:255',
            'collection_date' => 'nullable|date',
            'collection_location' => 'nullable|string|max:255',
            'collection_notes' => 'nullable|string',
            'collected_by_officer_id' => 'nullable|exists:users,id',
            'custodian' => 'nullable|exists:users,id',
            'case_id' => 'nullable|exists:cases,id',
            'photos.*' => 'nullable|file|mimes:jpeg,png,jpg,gif|max:2048',
            'documents.*' => 'nullable|file|mimes:pdf,doc,docx,xls,xlsx,txt|max:2048',
        ]);
        
        // Create new evidence record
        $evidence = new Evidence($validated);
        $evidence->created_by_user_id = Auth::id();
        $evidence->updated_by_user_id = Auth::id();
        
        // Initialize arrays for photos and documents
        $photos = [];
        $documents = [];
        
        // Handle photo uploads
        if ($request->hasFile('photos')) {
            foreach ($request->file('photos') as $photo) {
                $path = $photo->store('evidence/photos', 'public');
                $photos[] = [
                    'path' => $path,
                    'name' => $photo->getClientOriginalName(),
                    'uploaded_at' => now()->toDateTimeString(),
                    'uploaded_by' => Auth::id(),
                ];
            }
        }
        
        // Handle document uploads
        if ($request->hasFile('documents')) {
            foreach ($request->file('documents') as $document) {
                $path = $document->store('evidence/documents', 'public');
                $documents[] = [
                    'path' => $path,
                    'name' => $document->getClientOriginalName(),
                    'uploaded_at' => now()->toDateTimeString(),
                    'uploaded_by' => Auth::id(),
                ];
            }
        }
        
        // Set photos and documents
        $evidence->photos = $photos;
        $evidence->documents = $documents;
        
        // Initialize chain of custody
        $chainOfCustody = [];
        if ($request->filled('custodian')) {
            $chainOfCustody[] = [
                'date' => now()->toDateTimeString(),
                'from_officer_id' => $request->collected_by_officer_id,
                'to_officer_id' => $request->custodian,
                'reason' => 'Initial custody assignment',
                'notes' => 'Evidence registered in system',
                'recorded_by' => Auth::id(),
            ];
        }
        $evidence->chain_of_custody = $chainOfCustody;
        
        // Save the evidence
        $evidence->save();
        
        return redirect()->route('evidence.show', $evidence->id)
            ->with('success', 'Evidence record created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $evidence = $this->getFilteredEvidence()
            ->with(['collector', 'currentCustodian', 'case'])
            ->findOrFail($id);
            
        // Check if user can access this evidence's case station
        if ($evidence->case && !Auth::user()->canAccessStationData($evidence->case->station_id)) {
            return redirect()->route('evidence.index')
                ->with('error', 'You do not have permission to view this evidence.');
        }
        
        // Get officers for transfer modal
        $officers = $this->getFilteredOfficers()->get();
        
        return view('evidence.show', compact('evidence', 'officers'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $evidence = $this->getFilteredEvidence()->findOrFail($id);
        
        // Check if user can access this evidence's case station
        if ($evidence->case && !Auth::user()->canAccessStationData($evidence->case->station_id)) {
            return redirect()->route('evidence.index')
                ->with('error', 'You do not have permission to edit this evidence.');
        }
        
        // Get evidence types for dropdown
        $evidenceTypes = [
            'physical' => 'Physical',
            'digital' => 'Digital',
            'biological' => 'Biological',
            'documentary' => 'Documentary',
            'weapon' => 'Weapon',
            'drug' => 'Drug',
            'other' => 'Other',
        ];
        
        // Get statuses for dropdown
        $statuses = [
            'in_custody' => 'In Custody',
            'in_analysis' => 'In Analysis',
            'returned' => 'Returned',
            'destroyed' => 'Destroyed',
            'missing' => 'Missing',
            'transferred' => 'Transferred',
        ];
        
        // Get officers from accessible stations only
        $officers = $this->getFilteredOfficers()->get();
        
        return view('evidence.edit', compact('evidence', 'evidenceTypes', 'statuses', 'officers'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        // Find the evidence with station filtering
        $evidence = $this->getFilteredEvidence()->findOrFail($id);
        
        // Check if user can access this evidence's case station
        if ($evidence->case && !Auth::user()->canAccessStationData($evidence->case->station_id)) {
            return redirect()->route('evidence.index')
                ->with('error', 'You do not have permission to update this evidence.');
        }
        
        // Validate the request
        $validated = $request->validate([
            'tag_number' => 'required|string|max:255|unique:evidence,tag_number,' . $id,
            'evidence_type' => 'required|string',
            'description' => 'required|string',
            'status' => 'required|string',
            'condition' => 'nullable|string|max:255',
            'location' => 'nullable|string|max:255',
            'storage_location' => 'nullable|string|max:255',
            'collection_date' => 'nullable|date',
            'collection_location' => 'nullable|string|max:255',
            'collection_notes' => 'nullable|string',
            'collected_by_officer_id' => 'nullable|exists:users,id',
            'custodian' => 'nullable|exists:users,id',
            'case_id' => 'nullable|exists:cases,id',
            'photos.*' => 'nullable|file|mimes:jpeg,png,jpg,gif|max:2048',
            'documents.*' => 'nullable|file|mimes:pdf,doc,docx,xls,xlsx,txt|max:2048',
            'analysis_date' => 'nullable|date',
            'analysis_results' => 'nullable|string',
            'analyzed_by_officer_id' => 'nullable|exists:users,id',
        ]);
        
        // Update evidence record
        $evidence->fill($validated);
        $evidence->updated_by_user_id = Auth::id();
        
        // Handle photo uploads
        if ($request->hasFile('photos')) {
            // Get existing photos from the raw JSON to preserve structure
            $existingPhotosJson = $evidence->getRawOriginal('photos');
            
            // Handle different formats of existing photos
            if (empty($existingPhotosJson)) {
                $existingPhotos = [[]];
            } else {
                try {
                    $existingPhotos = json_decode($existingPhotosJson, true);
                    
                    // If it's not an array or is a string, initialize properly
                    if (!is_array($existingPhotos)) {
                        $existingPhotos = [[]];
                    } 
                    // If we don't have the empty first element, add it
                    else if (!isset($existingPhotos[0])) {
                        array_unshift($existingPhotos, []);
                    }
                } catch (\Exception $e) {
                    // If there's any error in decoding, start fresh
                    $existingPhotos = [[]];
                }
            }
            
            // Add new photos
            foreach ($request->file('photos') as $photo) {
                $path = $photo->store('evidence/photos', 'public');
                $existingPhotos[] = [
                    'path' => $path,
                    'name' => $photo->getClientOriginalName(),
                    'uploaded_at' => now()->toDateTimeString(),
                    'uploaded_by' => Auth::id(),
                ];
            }
            
            // Update the photos JSON
            $evidence->photos = $existingPhotos;
        }
        // If photos field exists but is not a file upload (might be from a form submission)
        else if ($request->has('photos') && is_array($request->photos)) {
            // Keep the existing photos unchanged
            // This prevents the photos field from being overwritten with form data
        }
        
        // Handle document uploads
        if ($request->hasFile('documents')) {
            // Get existing documents from the raw JSON to preserve structure
            $existingDocsJson = $evidence->getRawOriginal('documents');
            
            // Handle different formats of existing documents
            if (empty($existingDocsJson)) {
                $existingDocs = [[]];
            } else {
                try {
                    $existingDocs = json_decode($existingDocsJson, true);
                    
                    // If it's not an array or is a string, initialize properly
                    if (!is_array($existingDocs)) {
                        $existingDocs = [[]];
                    } 
                    // If we don't have the empty first element, add it
                    else if (!isset($existingDocs[0])) {
                        array_unshift($existingDocs, []);
                    }
                } catch (\Exception $e) {
                    // If there's any error in decoding, start fresh
                    $existingDocs = [[]];
                }
            }
            
            // Add new documents
            foreach ($request->file('documents') as $document) {
                $path = $document->store('evidence/documents', 'public');
                $existingDocs[] = [
                    'path' => $path,
                    'name' => $document->getClientOriginalName(),
                    'uploaded_at' => now()->toDateTimeString(),
                    'uploaded_by' => Auth::id(),
                ];
            }
            
            // Update the documents JSON
            $evidence->documents = $existingDocs;
        } 
        // If documents field exists but is not a file upload (might be from a form submission)
        else if ($request->has('documents') && is_array($request->documents)) {
            // Keep the existing documents unchanged
            // This prevents the documents field from being overwritten with form data
        }
        
        // Handle custodian change
        if ($request->filled('custodian') && $evidence->getOriginal('custodian') != $request->custodian) {
            $chainOfCustody = $evidence->chain_of_custody ?: [];
            $chainOfCustody[] = [
                'date' => now()->toDateTimeString(),
                'from_officer_id' => $evidence->getOriginal('custodian'),
                'to_officer_id' => $request->custodian,
                'reason' => 'Custody transfer during evidence update',
                'notes' => $request->input('transfer_notes', 'Updated via evidence edit form'),
                'recorded_by' => Auth::id(),
            ];
            $evidence->chain_of_custody = $chainOfCustody;
        }
        
        // Save the evidence
        $evidence->save();
        
        return redirect()->route('evidence.show', $evidence->id)
            ->with('success', 'Evidence record updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $evidence = $this->getFilteredEvidence()->findOrFail($id);
        
        // Check if user can access this evidence's case station
        if ($evidence->case && !Auth::user()->canAccessStationData($evidence->case->station_id)) {
            return redirect()->route('evidence.index')
                ->with('error', 'You do not have permission to delete this evidence.');
        }
        
        // Soft delete the evidence
        $evidence->delete();
        
        return redirect()->route('evidence.index')
            ->with('success', 'Evidence record deleted successfully.');
    }
    
    /**
     * Transfer custody of evidence to another officer.
     */
    public function transfer(Request $request, string $id)
    {
        // Validate the request
        $validated = $request->validate([
            'to_officer_id' => 'required|exists:users,id',
            'reason' => 'required|string',
            'notes' => 'nullable|string',
        ]);
        
        // Find the evidence with station filtering
        $evidence = $this->getFilteredEvidence()->findOrFail($id);
        
        // Check if target officer is accessible (same station or multi-station access)
        $targetOfficer = User::findOrFail($validated['to_officer_id']);
        if (!Auth::user()->canAccessStationData($targetOfficer->station_id)) {
            return redirect()->route('evidence.show', $id)
                ->with('error', 'You cannot transfer evidence to an officer from a different station.');
        }
        
        // Add to chain of custody
        $chainOfCustody = $evidence->chain_of_custody ?: [];
        $chainOfCustody[] = [
            'date' => now()->toDateTimeString(),
            'from_officer_id' => $evidence->custodian,
            'to_officer_id' => $validated['to_officer_id'],
            'reason' => $validated['reason'],
            'notes' => $validated['notes'],
            'recorded_by' => Auth::id(),
        ];
        
        // Update evidence
        $evidence->chain_of_custody = $chainOfCustody;
        $evidence->custodian = $validated['to_officer_id'];
        $evidence->updated_by_user_id = Auth::id();
        $evidence->save();
        
        return redirect()->route('evidence.show', $evidence->id)
            ->with('success', 'Evidence custody transferred successfully.');
    }
    
    /**
     * Remove a photo from evidence.
     */
    public function removePhoto(Request $request, string $id)
    {
        // Validate the request
        $request->validate([
            'photo_index' => 'required|integer|min:0',
        ]);
        
        // Find the evidence with station filtering
        $evidence = $this->getFilteredEvidence()->findOrFail($id);
        
        // Check if user can access this evidence's case station
        if ($evidence->case && !Auth::user()->canAccessStationData($evidence->case->station_id)) {
            return redirect()->route('evidence.index')
                ->with('error', 'You do not have permission to remove photos from this evidence.');
        }
        
        // Get photos
        $photos = $evidence->photos ?: [];
        
        // Check if the photo exists
        if (isset($photos[$request->photo_index])) {
            // Delete the file
            $photo = $photos[$request->photo_index];
            if (isset($photo['path'])) {
                Storage::disk('public')->delete($photo['path']);
            }
            
            // Remove from array
            array_splice($photos, $request->photo_index, 1);
            
            // Update evidence
            $evidence->photos = $photos;
            $evidence->updated_by_user_id = Auth::id();
            $evidence->save();
            
            return redirect()->route('evidence.show', $evidence->id)
                ->with('success', 'Photo removed successfully.');
        }
        
        return redirect()->route('evidence.show', $evidence->id)
            ->with('error', 'Photo not found.');
    }
    
    /**
     * Remove a document from evidence.
     */
    public function removeDocument(Request $request, string $id)
    {
        // Validate the request
        $request->validate([
            'document_index' => 'required|integer|min:0',
        ]);
        
        // Find the evidence with station filtering
        $evidence = $this->getFilteredEvidence()->findOrFail($id);
        
        // Check if user can access this evidence's case station
        if ($evidence->case && !Auth::user()->canAccessStationData($evidence->case->station_id)) {
            return redirect()->route('evidence.index')
                ->with('error', 'You do not have permission to remove documents from this evidence.');
        }
        
        // Get documents
        $documents = $evidence->documents ?: [];
        
        // Check if the document exists
        if (isset($documents[$request->document_index])) {
            // Delete the file
            $document = $documents[$request->document_index];
            if (isset($document['path'])) {
                Storage::disk('public')->delete($document['path']);
            }
            
            // Remove from array
            array_splice($documents, $request->document_index, 1);
            
            // Update evidence
            $evidence->documents = $documents;
            $evidence->updated_by_user_id = Auth::id();
            $evidence->save();
            
            return redirect()->route('evidence.show', $evidence->id)
                ->with('success', 'Document removed successfully.');
        }
        
        return redirect()->route('evidence.show', $evidence->id)
            ->with('error', 'Document not found.');
    }
}
