<?php

namespace App\Http\Controllers;

use App\Models\CaseEvidence;
use App\Models\EvidenceTransfer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class EvidenceTransferController extends Controller
{
    /**
     * Store a newly created transfer record.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'evidence_id' => 'required|exists:case_evidences,id',
            'from_person' => 'required|string|max:100',
            'to_person' => 'required|string|max:100',
            'transfer_date' => 'required|date',
            'transfer_time' => 'required',
            'reason' => 'required|string|max:50',
            'notes' => 'nullable|string',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
        
        try {
            // Create the transfer record
            $transfer = new EvidenceTransfer();
            $transfer->evidence_id = $request->evidence_id;
            $transfer->from_person = $request->from_person;
            $transfer->to_person = $request->to_person;
            $transfer->transfer_date = $request->transfer_date;
            $transfer->transfer_time = $request->transfer_time;
            $transfer->reason = $request->reason;
            $transfer->notes = $request->notes;
            $transfer->recorded_by_user_id = Auth::id();
            $transfer->save();
            
            // Update the evidence record with new custodian
            $evidence = CaseEvidence::findOrFail($request->evidence_id);
            $evidence->custodian = $request->to_person;
            $evidence->updated_by_user_id = Auth::id();
            
            // Update chain of custody
            $chainOfCustody = json_decode($evidence->chain_of_custody ?? '[]', true);
            if (!is_array($chainOfCustody)) {
                $chainOfCustody = [];
            }
            
            // Add new custody entry
            $chainOfCustody[] = [
                'date' => $request->transfer_date,
                'time' => $request->transfer_time,
                'from' => $request->from_person,
                'to' => $request->to_person,
                'reason' => $request->reason,
                'notes' => $request->notes,
                'recorded_by' => Auth::user()->name,
                'recorded_at' => now()->toDateTimeString(),
            ];
            
            $evidence->chain_of_custody = $chainOfCustody;
            $evidence->save();
            
            // Create a case update for the transfer
            $evidence->case->updates()->create([
                'update_type' => 'evidence',
                'description' => 'Evidence #' . $evidence->tag_number . ' transferred from ' . 
                    $request->from_person . ' to ' . $request->to_person,
                'officer_id' => Auth::id(),
                'update_date' => now(),
            ]);
            
            return redirect()->route('case-evidences.show', $evidence->id)
                ->with('success', 'Evidence transfer recorded successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error recording transfer: ' . $e->getMessage())
                ->withInput();
        }
    }
    
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = EvidenceTransfer::with('evidence');
        $evidence = null;
        
        // Filter by evidence if provided
        if ($request->has('evidence_id') && $request->evidence_id) {
            $query->where('evidence_id', $request->evidence_id);
            $evidence = CaseEvidence::findOrFail($request->evidence_id);
        }
        
        $transfers = $query->orderBy('transfer_date', 'desc')
            ->orderBy('transfer_time', 'desc')
            ->paginate(15);
        
        // Get all evidences for the filter dropdown
        $evidences = CaseEvidence::all();
        
        return view('evidence-transfers.index', compact('transfers', 'evidences', 'evidence'));
    }
    
    /**
     * Remove the specified transfer record.
     */
    public function destroy($id)
    {
        try {
            $transfer = EvidenceTransfer::findOrFail($id);
            $evidenceId = $transfer->evidence_id;
            
            // Delete the transfer record
            $transfer->delete();
            
            return redirect()->route('evidence-transfers.index', $evidenceId)
                ->with('success', 'Transfer record deleted successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error deleting transfer record: ' . $e->getMessage());
        }
    }
}
