<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\PoliceStation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class PoliceStationController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): View
    {
        $stations = PoliceStation::with('commander')->latest()->paginate(10);
        return view('stations.index', compact('stations'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(): View
    {
        $commanders = User::where('is_officer', true)->get();
        return view('stations.create', compact('commanders'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:50|unique:police_stations',
            'address' => 'required|string|max:255',
            'city' => 'required|string|max:100',
            'region' => 'required|string|max:100',
            'phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'description' => 'nullable|string',
            'is_headquarters' => 'boolean',
            'is_active' => 'boolean',
            'commander_id' => 'nullable|exists:users,id',
        ]);

        // If this is set as headquarters, unset any existing headquarters
        if ($request->boolean('is_headquarters')) {
            DB::table('police_stations')
                ->where('is_headquarters', true)
                ->update(['is_headquarters' => false]);
        }

        PoliceStation::create($validated);

        return redirect()->route('stations.index')
            ->with('success', 'Police station created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(PoliceStation $station): View
    {
        $station->load('commander', 'officers');
        return view('stations.show', compact('station'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(PoliceStation $station): View
    {
        $commanders = User::where('is_officer', true)->get();
        return view('stations.edit', compact('station', 'commanders'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, PoliceStation $station): RedirectResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:50|unique:police_stations,code,' . $station->id,
            'address' => 'required|string|max:255',
            'city' => 'required|string|max:100',
            'region' => 'required|string|max:100',
            'phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'description' => 'nullable|string',
            'is_headquarters' => 'boolean',
            'is_active' => 'boolean',
            'commander_id' => 'nullable|exists:users,id',
        ]);

        // If this is set as headquarters, unset any existing headquarters
        if ($request->boolean('is_headquarters') && !$station->is_headquarters) {
            DB::table('police_stations')
                ->where('is_headquarters', true)
                ->update(['is_headquarters' => false]);
        }

        $station->update($validated);

        return redirect()->route('stations.index')
            ->with('success', 'Police station updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(PoliceStation $station): RedirectResponse
    {
        // Check if station has officers or cases before deletion
        if ($station->officers()->count() > 0) {
            return redirect()->route('stations.index')
                ->with('error', 'Cannot delete station with assigned officers.');
        }

        $station->delete();

        return redirect()->route('stations.index')
            ->with('success', 'Police station deleted successfully.');
    }
}
