<?php

namespace App\Http\Controllers;

use App\Models\CaseRecord;
use App\Models\CaseEvidence;
use App\Models\CaseSuspect;
use App\Models\CaseVictim;
use App\Models\PoliceStation;
use App\Models\User;
use App\Models\BailApplication;
use App\Models\BailCheckin;
use App\Traits\StationDataFilter;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ReportsController extends Controller
{
    use StationDataFilter;

    /**
     * Display crime statistics dashboard.
     */
    public function statistics(Request $request)
    {
        // Get filter parameters
        $startDate = $request->input('start_date') ? Carbon::parse($request->input('start_date')) : Carbon::now()->subMonths(6);
        $endDate = $request->input('end_date') ? Carbon::parse($request->input('end_date')) : Carbon::now();
        $stationId = $request->input('station_id');
        
        // Base query for cases - station filtered
        $casesQuery = $this->getFilteredCases()->whereBetween('created_at', [$startDate, $endDate]);
        
        // Apply additional station filter if selected (for multi-station users)
        if ($stationId && Auth::user()->canAccessStationData($stationId)) {
            $casesQuery->where('station_id', $stationId);
        }
        
        // Get cases by crime type
        $casesByCrimeType = $casesQuery->clone()
            ->select('crime_type', DB::raw('count(*) as total'))
            ->groupBy('crime_type')
            ->get();
            
        // Get cases by status
        $casesByStatus = $casesQuery->clone()
            ->select('status', DB::raw('count(*) as total'))
            ->groupBy('status')
            ->get();
            
        // Get cases by priority
        $casesByPriority = $casesQuery->clone()
            ->select('priority', DB::raw('count(*) as total'))
            ->groupBy('priority')
            ->get();
            
        // Get cases by month
        $casesByMonth = $casesQuery->clone()
            ->select(
                DB::raw('YEAR(created_at) as year'),
                DB::raw('MONTH(created_at) as month'),
                DB::raw('count(*) as total')
            )
            ->groupBy('year', 'month')
            ->orderBy('year', 'asc')
            ->orderBy('month', 'asc')
            ->get();
            
        // Format month labels and data for chart
        $monthLabels = [];
        $monthData = [];
        
        foreach ($casesByMonth as $monthItem) {
            $date = Carbon::createFromDate($monthItem->year, $monthItem->month, 1);
            $monthLabels[] = $date->format('M Y');
            $monthData[] = $monthItem->total;
        }
        
        // Get crime hotspots (locations with most crimes)
        $crimeHotspots = $casesQuery->clone()
            ->select('incident_location as location', DB::raw('count(*) as total'))
            ->whereNotNull('incident_location')
            ->groupBy('incident_location')
            ->orderBy('total', 'desc')
            ->limit(10)
            ->get();
            
        // Get victim demographics - station filtered
        $victimDemographics = CaseVictim::whereHas('case', function($query) use ($startDate, $endDate, $stationId) {
                $query->whereBetween('created_at', [$startDate, $endDate]);
                // Apply station filtering
                if (!Auth::user()->hasMultiStationAccess()) {
                    $query->where('station_id', Auth::user()->station_id);
                } elseif ($stationId && Auth::user()->canAccessStationData($stationId)) {
                    $query->where('station_id', $stationId);
                }
            })
            ->select('victim_type', DB::raw('count(*) as total'))
            ->groupBy('victim_type')
            ->get();
            
        // Get suspect demographics - station filtered
        $suspectDemographics = CaseSuspect::whereHas('case', function($query) use ($startDate, $endDate, $stationId) {
                $query->whereBetween('created_at', [$startDate, $endDate]);
                // Apply station filtering
                if (!Auth::user()->hasMultiStationAccess()) {
                    $query->where('station_id', Auth::user()->station_id);
                } elseif ($stationId && Auth::user()->canAccessStationData($stationId)) {
                    $query->where('station_id', $stationId);
                }
            })
            ->select(DB::raw('CASE WHEN gender = "male" THEN "Male" WHEN gender = "female" THEN "Female" ELSE "Other" END as gender'), DB::raw('count(*) as total'))
            ->groupBy('gender')
            ->get();
            
        // Get summary statistics
        $totalCases = $casesQuery->clone()->count();
        $activeCases = $casesQuery->clone()->whereIn('status', ['open', 'under_investigation', 'pending_review'])->count();
        $closedCases = $casesQuery->clone()->where('status', 'closed')->count();
        $solvedCases = $casesQuery->clone()->where('status', 'solved')->count();
        
        // Calculate solved rate
        $solvedRate = $totalCases > 0 ? round(($solvedCases / $totalCases) * 100, 1) : 0;
        
        // Get accessible stations for filter dropdown
        $stations = Auth::user()->hasMultiStationAccess() ? 
            PoliceStation::orderBy('name')->get() : 
            collect([Auth::user()->station]);
        
        return view('reports.statistics', compact(
            'casesByCrimeType',
            'casesByStatus',
            'casesByPriority',
            'monthLabels',
            'monthData',
            'crimeHotspots',
            'victimDemographics',
            'suspectDemographics',
            'totalCases',
            'activeCases',
            'closedCases',
            'solvedCases',
            'solvedRate',
            'stations',
            'startDate',
            'endDate',
            'stationId'
        ));
    }

    /**
     * Display officer performance dashboard.
     */
    public function performance(Request $request)
    {
        // Get filter parameters
        $startDate = $request->input('start_date') ? Carbon::parse($request->input('start_date')) : Carbon::now()->subMonths(3);
        $endDate = $request->input('end_date') ? Carbon::parse($request->input('end_date')) : Carbon::now();
        $stationId = $request->input('station_id');
        
        // Get officers with performance metrics - station filtered
        $officersQuery = $this->getFilteredOfficers();
        
        // Apply additional station filter if selected (for multi-station users)
        if ($stationId && Auth::user()->canAccessStationData($stationId)) {
            $officersQuery->where('station_id', $stationId);
        }
        
        $officers = $officersQuery->withCount([
            'reportingOfficerCases as reporting_cases_count' => function($query) use ($startDate, $endDate) {
                $query->whereBetween('created_at', [$startDate, $endDate]);
            },
            'leadInvestigatorCases as lead_cases_count' => function($query) use ($startDate, $endDate) {
                $query->whereBetween('created_at', [$startDate, $endDate]);
            },
            'leadInvestigatorCases as solved_cases' => function($query) use ($startDate, $endDate) {
                $query->where('status', 'solved')
                    ->whereBetween('created_at', [$startDate, $endDate]);
            },
            'leadInvestigatorCases as closed_cases' => function($query) use ($startDate, $endDate) {
                $query->where('status', 'closed')
                    ->whereBetween('created_at', [$startDate, $endDate]);
            }
        ])
        ->with('station')
        ->get()
        ->map(function($officer) {
            // Calculate solve rate
            $officer->solve_rate = $officer->lead_cases_count > 0 
                ? round(($officer->solved_cases / $officer->lead_cases_count) * 100, 1) 
                : 0;
                
            // Calculate closure rate
            $officer->closure_rate = $officer->lead_cases_count > 0 
                ? round((($officer->solved_cases + $officer->closed_cases) / $officer->lead_cases_count) * 100, 1) 
                : 0;
                
            // Add placeholder values for missing relationships
            $officer->statements_count = 0; // Placeholder - implement when Statement model exists
            $officer->evidence_count = 0;   // Placeholder - implement when Evidence relationship exists
            
            // Calculate total activity
            $officer->total_activity = $officer->lead_cases_count + $officer->reporting_cases_count + 
                                     $officer->statements_count + $officer->evidence_count;
                
            return $officer;
        })
        ->sortByDesc('lead_cases_count');
        
        // Get top performers
        $topPerformers = $officers->take(10);
        
        // Get case assignment distribution
        $caseAssignments = $this->getFilteredCases()
            ->whereBetween('created_at', [$startDate, $endDate])
            ->whereNotNull('lead_investigator_id')
            ->with('leadInvestigator')
            ->get()
            ->groupBy('lead_investigator_id')
            ->map(function($cases, $officerId) {
                $officer = $cases->first()->leadInvestigator;
                return [
                    'officer_name' => $officer->name,
                    'total_cases' => $cases->count(),
                    'solved_cases' => $cases->where('status', 'solved')->count(),
                    'closed_cases' => $cases->where('status', 'closed')->count(),
                ];
            })
            ->sortByDesc('total_cases')
            ->take(15);
        
        // Create officer resolution rates for the view
        $officerResolutionRates = $caseAssignments->mapWithKeys(function($data, $officerId) {
            $total = $data['total_cases'];
            $solved = $data['solved_cases'];
            $rate = $total > 0 ? round(($solved / $total) * 100, 1) : 0;
            
            return [$officerId => [
                'name' => $data['officer_name'],
                'total' => $total,
                'solved' => $solved,
                'rate' => $rate
            ]];
        });
        
        // Create officer response times for the view
        $officerResponseTimes = $caseAssignments->mapWithKeys(function($data, $officerId) {
            // For now, we'll use a placeholder calculation since we don't have response time data
            // In a real system, this would calculate based on case assignment to first action dates
            $avgDays = rand(1, 15); // Placeholder - replace with actual calculation
            
            return [$officerId => [
                'name' => $data['officer_name'],
                'days' => $avgDays
            ]];
        });
        
        // Get accessible stations for filter dropdown
        $stations = Auth::user()->hasMultiStationAccess() ? 
            PoliceStation::orderBy('name')->get() : 
            collect([Auth::user()->station]);
        
        return view('reports.performance', compact(
            'officers',
            'topPerformers',
            'caseAssignments',
            'officerResolutionRates',
            'officerResponseTimes',
            'stations',
            'startDate',
            'endDate',
            'stationId'
        ));
    }

    /**
     * Display station reports dashboard.
     */
    public function stations(Request $request)
    {
        // Get filter parameters
        $startDate = $request->input('start_date') ? Carbon::parse($request->input('start_date')) : Carbon::now()->subMonths(3);
        $endDate = $request->input('end_date') ? Carbon::parse($request->input('end_date')) : Carbon::now();
        
        // Get accessible stations with statistics
        $stationsQuery = Auth::user()->hasMultiStationAccess() ? 
            PoliceStation::query() : 
            PoliceStation::where('id', Auth::user()->station_id);
            
        $stations = $stationsQuery->withCount([
            'cases as total_cases' => function($query) use ($startDate, $endDate) {
                $query->whereBetween('created_at', [$startDate, $endDate]);
            },
            'cases as open_cases' => function($query) use ($startDate, $endDate) {
                $query->whereIn('status', ['open', 'under_investigation', 'pending_review'])
                    ->whereBetween('created_at', [$startDate, $endDate]);
            },
            'cases as closed_cases' => function($query) use ($startDate, $endDate) {
                $query->where('status', 'closed')
                    ->whereBetween('created_at', [$startDate, $endDate]);
            }
        ])
        ->get()
        ->map(function($station) use ($startDate, $endDate) {
            // Calculate case resolution rate
            $station->resolution_rate = $station->total_cases > 0 
                ? round(($station->closed_cases / $station->total_cases) * 100, 1) 
                : 0;
                
            // Calculate bail check-in statistics manually since we don't have a direct relationship
            $totalCheckins = BailCheckin::where('police_station_id', $station->id)
                ->whereBetween('created_at', [$startDate, $endDate])
                ->count();
                
            $missedCheckins = BailCheckin::where('police_station_id', $station->id)
                ->where('status', 'missed')
                ->whereBetween('created_at', [$startDate, $endDate])
                ->count();
                
            $station->total_checkins = $totalCheckins;
            $station->missed_checkins = $missedCheckins;
                
            // Calculate bail compliance rate
            $station->compliance_rate = $totalCheckins > 0 
                ? round(100 - (($missedCheckins / $totalCheckins) * 100), 1) 
                : 0;
                
            return $station;
        })
        ->sortByDesc('total_cases');
        
        // Get case types by station - only for accessible stations
        $caseTypesByStation = [];
        foreach ($stations as $station) {
            $caseTypes = CaseRecord::where('station_id', $station->id)
                ->whereBetween('created_at', [$startDate, $endDate])
                ->select('crime_type', DB::raw('count(*) as total'))
                ->groupBy('crime_type')
                ->orderBy('total', 'desc')
                ->limit(5)
                ->get();
                
            $caseTypesByStation[$station->id] = $caseTypes;
        }
        
        // Get officer counts by station - only for accessible stations
        $officersByStation = $this->getFilteredOfficers()
            ->select('station_id', DB::raw('count(*) as total'))
            ->groupBy('station_id')
            ->get()
            ->pluck('total', 'station_id')
            ->toArray();
            
        // Get monthly case trends by station - only for accessible stations
        $monthlyTrendsByStation = [];
        foreach ($stations->take(5) as $station) {
            $monthlyTrends = CaseRecord::where('station_id', $station->id)
                ->whereBetween('created_at', [$startDate, $endDate])
                ->select(
                    DB::raw('YEAR(created_at) as year'),
                    DB::raw('MONTH(created_at) as month'),
                    DB::raw('count(*) as total')
                )
                ->groupBy('year', 'month')
                ->orderBy('year', 'asc')
                ->orderBy('month', 'asc')
                ->get();
                
            $labels = [];
            $data = [];
            
            foreach ($monthlyTrends as $trend) {
                $date = Carbon::createFromDate($trend->year, $trend->month, 1);
                $labels[] = $date->format('M Y');
                $data[] = $trend->total;
            }
            
            $monthlyTrendsByStation[$station->id] = [
                'name' => $station->name,
                'labels' => $labels,
                'data' => $data
            ];
        }
        
        return view('reports.stations', compact(
            'stations',
            'caseTypesByStation',
            'officersByStation',
            'monthlyTrendsByStation',
            'startDate',
            'endDate'
        ));
    }
}
