<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class RankPermissionMiddleware
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next, string $permission): Response
    {
        $user = auth()->user();
        
        if (!$user) {
            return redirect()->route('login');
        }

        // Admin users (non-officers) have full access
        if (!$user->is_officer) {
            return $next($request);
        }

        // Check if user has the required permission based on rank
        if (!$this->hasPermission($user, $permission)) {
            abort(403, 'Access denied. Insufficient rank privileges.');
        }

        return $next($request);
    }

    /**
     * Check if user has permission based on their rank
     */
    private function hasPermission($user, $permission): bool
    {
        $rank = strtolower($user->rank ?? '');
        
        // Define rank hierarchy (higher number = higher authority)
        $rankHierarchy = [
            'constable' => 1,
            'corporal' => 2,
            'sergeant' => 3,
            'inspector' => 4,
            'chief inspector' => 5,
            'superintendent' => 6,
            'deputy superintendent' => 7,
            'assistant superintendent' => 8,
            'assistant commissioner' => 9,
            'deputy commissioner' => 10,
            'commissioner' => 11,
        ];

        // Define permissions and required minimum rank levels
        $permissions = [
            // Station Management - Inspector and above
            'manage-stations' => 4,
            'create-stations' => 6, // Superintendent and above
            
            // Officer Management - Inspector and above
            'manage-officers' => 4,
            'create-officers' => 5, // Chief Inspector and above
            'batch-assign-officers' => 6, // Superintendent and above
            
            // Case Management - All officers can view, higher ranks can manage
            'view-cases' => 1, // All officers
            'create-cases' => 2, // Corporal and above
            'edit-cases' => 3, // Sergeant and above
            'delete-cases' => 5, // Chief Inspector and above
            'close-cases' => 4, // Inspector and above
            
            // Evidence Management - Sergeant and above
            'manage-evidence' => 3,
            'transfer-evidence' => 4, // Inspector and above
            'delete-evidence' => 5, // Chief Inspector and above
            
            // Statements - Corporal and above can create, Sergeant and above can verify
            'create-statements' => 2,
            'verify-statements' => 3,
            'edit-statements' => 4,
            
            // Suspects and Victims - Corporal and above
            'manage-suspects' => 2,
            'manage-victims' => 2,
            'update-arrest-status' => 3, // Sergeant and above
            'update-charges' => 4, // Inspector and above
            
            // Bail Management - Inspector and above
            'manage-bail' => 4,
            'approve-bail' => 5, // Chief Inspector and above
            'bail-checkins' => 2, // Corporal and above can record check-ins
            
            // Reports and Analytics - Sergeant and above
            'view-reports' => 3,
            'advanced-reports' => 4, // Inspector and above
            'performance-reports' => 5, // Chief Inspector and above
            
            // System Settings - Superintendent and above
            'manage-settings' => 6,
            'manage-users' => 7, // Assistant Commissioner and above
            'system-backup' => 8, // Deputy Commissioner and above
        ];

        $userRankLevel = $rankHierarchy[$rank] ?? 0;
        $requiredLevel = $permissions[$permission] ?? 99; // Default to highest level if permission not found

        return $userRankLevel >= $requiredLevel;
    }
}
