<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class BailApplication extends Model
{
    use HasFactory, SoftDeletes;
    
    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'case_id',
        'suspect_id',
        'application_number',
        'application_date',
        'status',
        'approval_date',
        'expiry_date',
        'amount',
        'surety_name',
        'surety_contact',
        'surety_address',
        'surety_relationship',
        'surety_id_type',
        'surety_id_number',
        'reason',
        'notes',
        'approved_by',
        'created_by',
        'updated_by',
    ];
    
    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'application_date' => 'date',
        'approval_date' => 'date',
        'expiry_date' => 'date',
        'amount' => 'decimal:2',
    ];
    
    /**
     * Get the case associated with the bail application.
     */
    public function case(): BelongsTo
    {
        return $this->belongsTo(CaseRecord::class, 'case_id');
    }
    
    /**
     * Get the suspect associated with the bail application.
     */
    public function suspect(): BelongsTo
    {
        return $this->belongsTo(CaseSuspect::class, 'suspect_id');
    }
    
    /**
     * Get the user who approved the bail application.
     */
    public function approvedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approved_by');
    }
    
    /**
     * Get the user who created the bail application.
     */
    public function createdByUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }
    
    /**
     * Get the user who last updated the bail application.
     */
    public function updatedByUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by');
    }
    
    /**
     * Get the conditions for the bail application.
     */
    public function conditions(): HasMany
    {
        return $this->hasMany(BailCondition::class);
    }
    
    /**
     * Get the check-ins for the bail application.
     */
    public function checkins(): HasMany
    {
        return $this->hasMany(BailCheckin::class);
    }
    
    /**
     * Get the status badge HTML.
     */
    public function getStatusBadgeAttribute(): string
    {
        return match($this->status) {
            'pending' => '<span class="badge badge-warning">Pending</span>',
            'approved' => '<span class="badge badge-success">Approved</span>',
            'rejected' => '<span class="badge badge-danger">Rejected</span>',
            'revoked' => '<span class="badge badge-dark">Revoked</span>',
            default => '<span class="badge badge-secondary">Unknown</span>',
        };
    }
    
    /**
     * Check if the bail application is active (approved and not expired).
     */
    public function getIsActiveAttribute(): bool
    {
        return $this->status === 'approved' && 
               ($this->expiry_date === null || $this->expiry_date->isFuture());
    }
}
