<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class BailCheckin extends Model
{
    use HasFactory;
    
    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'bail_application_id',
        'suspect_id',
        'police_station_id',
        'checkin_time',
        'scheduled_time',
        'is_present',
        'status',
        'notes',
        'recorded_by',
        'created_by',
        'updated_by',
    ];
    
    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'checkin_time' => 'datetime',
        'scheduled_time' => 'datetime',
        'is_present' => 'boolean',
    ];
    
    /**
     * Get the bail application associated with the check-in.
     */
    public function bailApplication(): BelongsTo
    {
        return $this->belongsTo(BailApplication::class);
    }
    
    /**
     * Get the suspect associated with the check-in.
     */
    public function suspect(): BelongsTo
    {
        return $this->belongsTo(CaseSuspect::class, 'suspect_id');
    }
    
    /**
     * Get the police station where the check-in occurred.
     */
    public function policeStation(): BelongsTo
    {
        return $this->belongsTo(PoliceStation::class);
    }
    
    /**
     * Get the user who recorded the check-in.
     */
    public function recordedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'recorded_by');
    }
    
    /**
     * Get the user who created the check-in record.
     */
    public function createdByUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }
    
    /**
     * Get the user who last updated the check-in record.
     */
    public function updatedByUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by');
    }
    
    /**
     * Get the status badge HTML.
     */
    public function getStatusBadgeAttribute(): string
    {
        return match($this->status) {
            'scheduled' => '<span class="badge badge-info">Scheduled</span>',
            'present' => '<span class="badge badge-success">Present</span>',
            'missed' => '<span class="badge badge-danger">Missed</span>',
            'excused' => '<span class="badge badge-warning">Excused</span>',
            default => '<span class="badge badge-secondary">Unknown</span>',
        };
    }
    
    /**
     * Check if the check-in is scheduled for today.
     */
    public function getIsScheduledForTodayAttribute(): bool
    {
        return $this->scheduled_time && $this->scheduled_time->isToday();
    }
    
    /**
     * Check if the check-in is overdue.
     */
    public function getIsOverdueAttribute(): bool
    {
        return $this->status === 'scheduled' && 
               $this->scheduled_time && 
               $this->scheduled_time->isPast();
    }
}
