<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class CaseEvidence extends Model
{
    use HasFactory, SoftDeletes;
    
    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'case_evidences';
    
    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'case_id',
        'tag_number',
        'name',
        'description',
        'evidence_type',
        'subtype',
        'collection_datetime',
        'collection_time',
        'collection_location',
        'collection_details',
        'collected_by_officer_id',
        'storage_location',
        'status',
        'condition',
        'custodian',
        'is_sensitive',
        'photo_path',
        'photos',
        'documents',
        'chain_of_custody',
        'analysis_results',
        'analyzed_by_officer_id',
        'analyzed_at',
        'is_admissible',
        'admissibility_notes',
        'notes',
        'created_by_user_id',
        'updated_by_user_id',
    ];
    
    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'collection_datetime' => 'datetime',
        'collection_time' => 'datetime',
        'is_sensitive' => 'boolean',
        'analyzed_at' => 'datetime',
        'is_admissible' => 'boolean',
        'photos' => 'array',
        'documents' => 'array',
        'chain_of_custody' => 'array',
    ];
    
    /**
     * Get the case that owns the evidence.
     */
    public function case(): BelongsTo
    {
        return $this->belongsTo(CaseRecord::class, 'case_id');
    }
    
    /**
     * Get the officer who collected the evidence.
     */
    public function collector(): BelongsTo
    {
        return $this->belongsTo(User::class, 'collected_by_officer_id');
    }
    
    /**
     * Alias for collector() to maintain compatibility.
     */
    public function collectedBy(): BelongsTo
    {
        return $this->collector();
    }
    
    /**
     * Get the officer who analyzed the evidence.
     */
    public function analyzedByOfficer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'analyzed_by_officer_id');
    }
    
    /**
     * Get the current custodian of the evidence.
     * This uses the custodian field which may contain a user ID.
     */
    public function currentCustodian(): BelongsTo
    {
        return $this->belongsTo(User::class, 'custodian');
    }
    
    /**
     * Get the evidence type label.
     */
    public function getEvidenceTypeLabelAttribute(): string
    {
        $typeMap = [
            'physical' => 'Physical',
            'digital' => 'Digital',
            'documentary' => 'Documentary',
            'biological' => 'Biological',
            'forensic' => 'Forensic',
            'other' => 'Other',
        ];
        
        return $typeMap[$this->evidence_type] ?? ucfirst($this->evidence_type);
    }
    
    /**
     * Get the status label.
     */
    public function getStatusLabelAttribute(): string
    {
        $statusMap = [
            'collected' => 'Collected',
            'in_storage' => 'In Storage',
            'in_analysis' => 'In Analysis',
            'analyzed' => 'Analyzed',
            'in_court' => 'In Court',
            'returned' => 'Returned',
            'destroyed' => 'Destroyed',
        ];
        
        return $statusMap[$this->status] ?? ucfirst(str_replace('_', ' ', $this->status));
    }
    
    /**
     * Get the status badge class.
     */
    public function getStatusBadgeClassAttribute(): string
    {
        $badgeMap = [
            'collected' => 'badge-info',
            'in_storage' => 'badge-primary',
            'in_analysis' => 'badge-warning',
            'analyzed' => 'badge-success',
            'in_court' => 'badge-secondary',
            'returned' => 'badge-secondary',
            'destroyed' => 'badge-danger',
        ];
        
        return $badgeMap[$this->status] ?? 'badge-dark';
    }
    
    /**
     * Get the user who created the evidence record.
     */
    public function createdByUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by_user_id');
    }
    
    /**
     * Get the user who last updated the evidence record.
     */
    public function updatedByUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by_user_id');
    }
    
    /**
     * Get the photos attribute with fallback for string values.
     */
    public function getPhotosAttribute($value)
    {
        if (is_string($value) && !is_array(json_decode($value, true))) {
            return [];
        }
        return $this->castAttribute('photos', $value);
    }
    
    /**
     * Get the documents attribute with fallback for string values.
     */
    public function getDocumentsAttribute($value)
    {
        if (is_string($value) && !is_array(json_decode($value, true))) {
            return [];
        }
        return $this->castAttribute('documents', $value);
    }
    
    /**
     * Get the chain_of_custody attribute with fallback for string values.
     */
    public function getChainOfCustodyAttribute($value)
    {
        if (is_string($value) && !is_array(json_decode($value, true))) {
            return [];
        }
        return $this->castAttribute('chain_of_custody', $value);
    }
}
