<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class CaseRecord extends Model
{
    use HasFactory, SoftDeletes;
    
    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'cases';
    
    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'case_number',
        'title',
        'description',
        'station_id',
        'reporting_officer_id',
        'lead_investigator_id',
        'incident_date',
        'incident_time',
        'incident_location',
        'incident_coordinates',
        'crime_type',
        'crime_category',
        'status',
        'priority',
        'reported_date',
        'action_taken',
        'recommendations',
        'closed_date',
        'closed_by',
        'closing_remarks',
        'is_sensitive',
    ];
    
    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'incident_date' => 'date',
        'incident_time' => 'datetime',
        'reported_date' => 'date',
        'closed_date' => 'date',
        'is_sensitive' => 'boolean',
    ];
    
    /**
     * Get the police station that the case belongs to.
     */
    public function station(): BelongsTo
    {
        return $this->belongsTo(PoliceStation::class, 'station_id');
    }
    
    /**
     * Get the reporting officer for the case.
     */
    public function reportingOfficer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'reporting_officer_id');
    }
    
    /**
     * Get the lead investigator for the case.
     */
    public function leadInvestigator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'lead_investigator_id');
    }
    
    /**
     * Get the statements for the case.
     */
    public function statements(): HasMany
    {
        return $this->hasMany(CaseStatement::class, 'case_id');
    }
    
    /**
     * Get the evidence for the case.
     */
    public function evidence(): HasMany
    {
        return $this->hasMany(CaseEvidence::class, 'case_id');
    }
    
    /**
     * Alias for evidence() to maintain backward compatibility.
     */
    public function evidences(): HasMany
    {
        return $this->evidence();
    }
    
    /**
     * Get the suspects for the case.
     */
    public function suspects(): HasMany
    {
        return $this->hasMany(CaseSuspect::class, 'case_id');
    }
    
    /**
     * Get the victims for the case.
     */
    public function victims(): HasMany
    {
        return $this->hasMany(CaseVictim::class, 'case_id');
    }
    
    /**
     * Get the updates for the case.
     */
    public function updates(): HasMany
    {
        return $this->hasMany(CaseUpdate::class, 'case_id');
    }
    
    /**
     * Get the status label.
     */
    public function getStatusLabelAttribute(): string
    {
        $statusMap = [
            'open' => 'Open',
            'under_investigation' => 'Under Investigation',
            'pending_review' => 'Pending Review',
            'closed' => 'Closed',
            'reopened' => 'Reopened',
            'archived' => 'Archived',
        ];
        
        return $statusMap[$this->status] ?? $this->status;
    }
    
    /**
     * Get the priority label.
     */
    public function getPriorityLabelAttribute(): string
    {
        $priorityMap = [
            'low' => 'Low',
            'medium' => 'Medium',
            'high' => 'High',
            'critical' => 'Critical',
        ];
        
        return $priorityMap[$this->priority] ?? $this->priority;
    }
    
    /**
     * Get the priority badge class.
     */
    public function getPriorityBadgeClassAttribute(): string
    {
        $priorityClassMap = [
            'low' => 'badge-info',
            'medium' => 'badge-primary',
            'high' => 'badge-warning',
            'critical' => 'badge-danger',
        ];
        
        return $priorityClassMap[$this->priority] ?? 'badge-secondary';
    }
    
    /**
     * Get the status badge class.
     */
    public function getStatusBadgeClassAttribute(): string
    {
        $statusClassMap = [
            'open' => 'badge-success',
            'under_investigation' => 'badge-primary',
            'pending_review' => 'badge-info',
            'closed' => 'badge-secondary',
            'reopened' => 'badge-warning',
            'archived' => 'badge-dark',
        ];
        
        return $statusClassMap[$this->status] ?? 'badge-secondary';
    }
    
    /**
     * Get the user who created the case record.
     */
    public function createdByUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by_user_id');
    }
    
    /**
     * Get the user who last updated the case record.
     */
    public function updatedByUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by_user_id');
    }
}
