<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class CaseStatement extends Model
{
    use HasFactory, SoftDeletes;
    
    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'case_id',
        'statement_type',
        'given_by_name',
        'given_by_contact',
        'given_by_address',
        'given_by_id_type',
        'given_by_id_number',
        'recorded_by_officer_id',
        'statement_date',
        'statement_content',
        'location_taken',
        'is_sworn',
        'is_signed',
        'signature_file_path',
        'audio_recording_path',
        'officer_remarks',
        'is_verified',
        'verified_by_officer_id',
        'verified_at',
    ];
    
    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'statement_date' => 'datetime',
        'is_sworn' => 'boolean',
        'is_signed' => 'boolean',
        'is_verified' => 'boolean',
        'verified_at' => 'datetime',
    ];
    
    /**
     * Get the case that owns the statement.
     */
    public function case(): BelongsTo
    {
        return $this->belongsTo(CaseRecord::class, 'case_id');
    }
    
    /**
     * Get the officer who recorded the statement.
     */
    public function recordedByOfficer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'recorded_by_officer_id');
    }
    
    /**
     * Alias for recordedByOfficer() to maintain compatibility.
     */
    public function recordingOfficer(): BelongsTo
    {
        return $this->recordedByOfficer();
    }
    
    /**
     * Get the officer who verified the statement.
     */
    public function verifiedByOfficer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'verified_by_officer_id');
    }
    
    /**
     * Get the statement type label.
     */
    public function getStatementTypeLabelAttribute(): string
    {
        $typeMap = [
            'victim' => 'Victim',
            'witness' => 'Witness',
            'suspect' => 'Suspect',
            'officer' => 'Officer',
            'expert' => 'Expert',
        ];
        
        return $typeMap[$this->statement_type] ?? $this->statement_type;
    }
}
