<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class CaseSuspect extends Model
{
    use HasFactory, SoftDeletes;
    
    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'case_id',
        'full_name',
        'alias',
        'gender',
        'date_of_birth',
        'age',
        'nationality',
        'id_type',
        'id_number',
        'address',
        'phone',
        'email',
        'physical_description',
        'photo_path',
        'relationship_to_victim',
        'alleged_involvement',
        'is_arrested',
        'arrest_date',
        'arrest_location',
        'arrested_by_officer_id',
        'is_charged',
        'charges',
        'charge_date',
        'is_in_custody',
        'custody_location',
        'criminal_history',
        'officer_notes',
    ];
    
    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'date_of_birth' => 'date',
        'is_arrested' => 'boolean',
        'arrest_date' => 'datetime',
        'is_charged' => 'boolean',
        'charge_date' => 'datetime',
        'is_in_custody' => 'boolean',
    ];
    
    /**
     * Get the case that owns the suspect.
     */
    public function case(): BelongsTo
    {
        return $this->belongsTo(CaseRecord::class, 'case_id');
    }
    
    /**
     * Get the officer who arrested the suspect.
     */
    public function arrestedByOfficer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'arrested_by_officer_id');
    }
    
    /**
     * Get the suspect's full name with alias if available.
     */
    public function getFullNameWithAliasAttribute(): string
    {
        return $this->full_name . ($this->alias ? ' ("' . $this->alias . '")' : '');
    }
    
    /**
     * Get the suspect's age, calculated from date of birth if available.
     */
    public function getCalculatedAgeAttribute(): ?int
    {
        if ($this->date_of_birth) {
            return $this->date_of_birth->age;
        }
        
        return $this->age;
    }
    
    /**
     * Get the status updates for the suspect.
     */
    public function statusUpdates(): HasMany
    {
        return $this->hasMany(SuspectStatusUpdate::class, 'suspect_id');
    }
    
    /**
     * Get the bail applications for the suspect.
     */
    public function bailApplications(): HasMany
    {
        return $this->hasMany(BailApplication::class, 'suspect_id');
    }
}
