<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class CaseUpdate extends Model
{
    use HasFactory, SoftDeletes;
    
    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'case_id',
        'update_type',
        'description',
        'officer_id',
        'update_date',
        'location',
        'action_taken',
        'results',
        'next_steps',
        'is_significant',
        'requires_followup',
        'followup_date',
        'followup_action',
        'is_internal_only',
        'attachment_path',
    ];
    
    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'update_date' => 'datetime',
        'is_significant' => 'boolean',
        'requires_followup' => 'boolean',
        'followup_date' => 'datetime',
        'is_internal_only' => 'boolean',
    ];
    
    /**
     * Get the case that owns the update.
     */
    public function case(): BelongsTo
    {
        return $this->belongsTo(CaseRecord::class, 'case_id');
    }
    
    /**
     * Get the officer who created the update.
     */
    public function officer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'officer_id');
    }
    
    /**
     * Get the update type label.
     */
    public function getUpdateTypeLabelAttribute(): string
    {
        $typeMap = [
            'investigation' => 'Investigation',
            'evidence' => 'Evidence',
            'arrest' => 'Arrest',
            'interview' => 'Interview',
            'court' => 'Court Proceedings',
            'witness' => 'Witness Contact',
            'victim' => 'Victim Contact',
            'suspect' => 'Suspect Update',
            'general' => 'General Update',
        ];
        
        return $typeMap[$this->update_type] ?? $this->update_type;
    }
    
    /**
     * Get the update type badge class.
     */
    public function getUpdateTypeBadgeClassAttribute(): string
    {
        $typeClassMap = [
            'investigation' => 'badge-primary',
            'evidence' => 'badge-info',
            'arrest' => 'badge-danger',
            'interview' => 'badge-secondary',
            'court' => 'badge-dark',
            'witness' => 'badge-light',
            'victim' => 'badge-warning',
            'suspect' => 'badge-danger',
            'general' => 'badge-secondary',
        ];
        
        return $typeClassMap[$this->update_type] ?? 'badge-secondary';
    }
    
    /**
     * Check if the followup is overdue.
     */
    public function getIsFollowupOverdueAttribute(): bool
    {
        if (!$this->requires_followup || !$this->followup_date) {
            return false;
        }
        
        return $this->followup_date->isPast();
    }
}
