<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Log;

class CaseVictim extends Model
{
    use HasFactory, SoftDeletes;
    
    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'case_victims';
    
    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'case_id',
        'victim_type',
        'full_name',
        'organization_name',
        'organization_type',
        'registration_number',
        'contact_person_name',
        'contact_person_role',
        'gender',
        'date_of_birth',
        'age',
        'nationality',
        'id_type',
        'id_number',
        'address',
        'phone',
        'email',
        'occupation',
        'physical_description',
        'photo_path',
        'evidence_photos',
        'injuries_sustained',
        'injury_status',
        'medical_report',
        'medical_facility',
        'relationship_to_suspect',
        'is_vulnerable',
        'vulnerability_details',
        'requires_protection',
        'protection_details',
        'incident_date',
        'incident_location',
        'incident_description',
        'is_deceased',
        'time_of_death',
        'cause_of_death',
        'officer_notes',
        'created_by_user_id',
        'updated_by_user_id',
    ];
    
    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'date_of_birth' => 'date',
        'incident_date' => 'datetime',
        'is_vulnerable' => 'boolean',
        'requires_protection' => 'boolean',
        'is_deceased' => 'boolean',
        'time_of_death' => 'datetime',
        'evidence_photos' => 'array',
    ];
    
    /**
     * Get the case that owns the victim.
     */
    public function case(): BelongsTo
    {
        return $this->belongsTo(CaseRecord::class, 'case_id');
    }
    
    /**
     * Get the victim's calculated age, from date of birth if available.
     */
    public function getCalculatedAgeAttribute(): ?int
    {
        if ($this->date_of_birth) {
            return $this->date_of_birth->age;
        }
        
        return $this->age;
    }
    
    /**
     * Get the victim status label.
     */
    public function getStatusLabelAttribute(): string
    {
        if ($this->is_deceased) {
            return 'Deceased';
        } elseif ($this->injury_status && $this->injury_status != 'none') {
            $statusMap = [
                'minor' => 'Minor Injuries',
                'moderate' => 'Moderate Injuries',
                'severe' => 'Severe Injuries',
                'critical' => 'Critical Injuries',
            ];
            return $statusMap[$this->injury_status] ?? ucfirst($this->injury_status);
        } elseif ($this->requires_protection) {
            return 'Protected';
        } elseif ($this->is_vulnerable) {
            return 'Vulnerable';
        }
        
        return 'No Injuries';
    }
    
    /**
     * Get the victim status badge class.
     */
    public function getStatusBadgeClassAttribute(): string
    {
        if ($this->is_deceased) {
            return 'badge-danger';
        } elseif ($this->injury_status) {
            $badgeMap = [
                'none' => 'badge-secondary',
                'minor' => 'badge-info',
                'moderate' => 'badge-primary',
                'severe' => 'badge-warning',
                'critical' => 'badge-danger',
            ];
            return $badgeMap[$this->injury_status] ?? 'badge-secondary';
        } elseif ($this->requires_protection) {
            return 'badge-warning';
        } elseif ($this->is_vulnerable) {
            return 'badge-info';
        }
        
        return 'badge-secondary';
    }
    
    /**
     * Get the user who created the victim record.
     */
    public function createdByUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by_user_id');
    }
    
    /**
     * Get the user who last updated the victim record.
     */
    public function updatedByUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by_user_id');
    }
    
    /**
     * Get the protection officer assigned to the victim.
     */
    public function protectionOfficer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'protection_officer_id');
    }
    
    /**
     * Get the victim type label.
     */
    public function getVictimTypeLabelAttribute(): string
    {
        return $this->victim_type === 'individual' ? 'Individual' : 'Organization';
    }
    
    /**
     * Get the display name based on victim type.
     */
    public function getDisplayNameAttribute(): string
    {
        return $this->victim_type === 'individual' ? $this->full_name : $this->organization_name;
    }
    
    /**
     * Get the evidence photos as an array.
     */
    public function getEvidencePhotosAttribute($value)
    {
        if (empty($value)) {
            return [];
        }
        
        if (is_string($value)) {
            // Remove extra quotes if they exist (handles "[]") case
            if (substr($value, 0, 1) === '"' && substr($value, -1) === '"') {
                $value = substr($value, 1, -1);
            }
            
            try {
                return json_decode($value, true) ?: [];
            } catch (\Exception $e) {
                Log::error('Error decoding evidence_photos JSON: ' . $e->getMessage());
                return [];
            }
        }
        
        return $value;
    }
}
