<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Log;

class Evidence extends Model
{
    use HasFactory, SoftDeletes;
    
    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'tag_number',
        'evidence_type',
        'description',
        'status',
        'condition',
        'location',
        'storage_location',
        'collection_date',
        'collection_location',
        'collection_notes',
        'collected_by_officer_id',
        'custodian',
        'analyzed_by_officer_id',
        'analysis_date',
        'analysis_results',
        'photos',
        'documents',
        'chain_of_custody',
        'created_by_user_id',
        'updated_by_user_id',
        'case_id',
    ];
    
    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'photos' => 'array',
        'documents' => 'array',
        'chain_of_custody' => 'array',
        'collection_date' => 'datetime',
        'analysis_date' => 'datetime',
    ];
    
    /**
     * Get the photos as an array.
     */
    public function getPhotosAttribute($value)
    {
        if (empty($value)) {
            return [];
        }
        
        if (is_string($value)) {
            try {
                $photos = json_decode($value, true) ?: [];
                
                // Filter out the empty first element if it exists
                if (isset($photos[0]) && empty($photos[0])) {
                    $photos = array_filter($photos, function($key) {
                        return $key !== 0;
                    }, ARRAY_FILTER_USE_KEY);
                    
                    // Re-index array to start from 0
                    $photos = array_values($photos);
                }
                
                return $photos;
            } catch (\Exception $e) {
                Log::error('Error decoding photos JSON: ' . $e->getMessage());
                return [];
            }
        }
        
        return $value;
    }
    
    /**
     * Get the documents as an array.
     */
    public function getDocumentsAttribute($value)
    {
        if (empty($value)) {
            return [];
        }
        
        if (is_string($value)) {
            try {
                $documents = json_decode($value, true) ?: [];
                
                // Filter out the empty first element if it exists
                if (isset($documents[0]) && empty($documents[0])) {
                    $documents = array_filter($documents, function($key) {
                        return $key !== 0;
                    }, ARRAY_FILTER_USE_KEY);
                    
                    // Re-index array to start from 0
                    $documents = array_values($documents);
                }
                
                return $documents;
            } catch (\Exception $e) {
                Log::error('Error decoding documents JSON: ' . $e->getMessage());
                return [];
            }
        }
        
        return $value;
    }
    
    /**
     * Get the chain of custody as an array.
     */
    public function getChainOfCustodyAttribute($value)
    {
        if (empty($value)) {
            return [];
        }
        
        if (is_string($value)) {
            try {
                return json_decode($value, true) ?: [];
            } catch (\Exception $e) {
                Log::error('Error decoding chain_of_custody JSON: ' . $e->getMessage());
                return [];
            }
        }
        
        return $value;
    }
    
    /**
     * Get the case that owns the evidence.
     */
    public function case(): BelongsTo
    {
        return $this->belongsTo(CaseRecord::class, 'case_id');
    }
    
    /**
     * Get the officer who collected the evidence.
     */
    public function collector(): BelongsTo
    {
        return $this->belongsTo(User::class, 'collected_by_officer_id');
    }
    
    /**
     * Alias for collector() to maintain compatibility.
     */
    public function collectedBy(): BelongsTo
    {
        return $this->collector();
    }
    
    /**
     * Get the officer who analyzed the evidence.
     */
    public function analyzedByOfficer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'analyzed_by_officer_id');
    }
    
    /**
     * Get the current custodian of the evidence.
     */
    public function currentCustodian(): BelongsTo
    {
        return $this->belongsTo(User::class, 'custodian');
    }
    
    /**
     * Get the user who created the evidence record.
     */
    public function createdByUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by_user_id');
    }
    
    /**
     * Get the user who last updated the evidence record.
     */
    public function updatedByUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by_user_id');
    }
    
    /**
     * Get the evidence type label.
     */
    public function getEvidenceTypeLabelAttribute(): string
    {
        $types = [
            'physical' => 'Physical',
            'digital' => 'Digital',
            'biological' => 'Biological',
            'documentary' => 'Documentary',
            'weapon' => 'Weapon',
            'drug' => 'Drug',
            'other' => 'Other',
        ];
        
        return $types[$this->evidence_type] ?? ucfirst($this->evidence_type);
    }
    
    /**
     * Get the status label.
     */
    public function getStatusLabelAttribute(): string
    {
        $statuses = [
            'in_custody' => 'In Custody',
            'in_analysis' => 'In Analysis',
            'returned' => 'Returned',
            'destroyed' => 'Destroyed',
            'missing' => 'Missing',
            'transferred' => 'Transferred',
        ];
        
        return $statuses[$this->status] ?? ucfirst($this->status);
    }
    
    /**
     * Get the status badge class.
     */
    public function getStatusBadgeClassAttribute(): string
    {
        $classes = [
            'in_custody' => 'badge-primary',
            'in_analysis' => 'badge-info',
            'returned' => 'badge-success',
            'destroyed' => 'badge-danger',
            'missing' => 'badge-warning',
            'transferred' => 'badge-secondary',
        ];
        
        return $classes[$this->status] ?? 'badge-secondary';
    }
}
