<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class EvidenceTransfer extends Model
{
    use HasFactory;
    
    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'evidence_id',
        'from_person',
        'to_person',
        'transfer_date',
        'transfer_time',
        'reason',
        'notes',
        'recorded_by_user_id',
    ];
    
    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'transfer_date' => 'date',
    ];
    
    /**
     * Get the evidence that was transferred.
     */
    public function evidence(): BelongsTo
    {
        return $this->belongsTo(CaseEvidence::class, 'evidence_id');
    }
    
    /**
     * Get the user who recorded the transfer.
     */
    public function recordedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'recorded_by_user_id');
    }
    
    /**
     * Get the reason label.
     */
    public function getReasonLabelAttribute(): string
    {
        $reasonMap = [
            'analysis' => 'For Analysis',
            'storage' => 'For Storage',
            'court' => 'For Court Proceedings',
            'return' => 'Return to Owner',
            'disposal' => 'For Disposal',
            'other' => 'Other',
        ];
        
        return $reasonMap[$this->reason] ?? ucfirst($this->reason);
    }
    
    /**
     * Get the formatted transfer date and time.
     */
    public function getFormattedDateTimeAttribute(): string
    {
        return $this->transfer_date->format('d M Y') . ' at ' . $this->transfer_time->format('h:i A');
    }
}
