<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::table('case_evidence', function (Blueprint $table) {
            // Rename the table to plural form for consistency
            Schema::rename('case_evidence', 'case_evidences');
        });
        
        Schema::table('case_evidences', function (Blueprint $table) {
            // Rename fields to match our views
            $table->renameColumn('evidence_number', 'tag_number');
            $table->renameColumn('type', 'evidence_type');
            $table->renameColumn('collection_date', 'collection_datetime');
            
            // Add missing fields from our views
            $table->time('collection_time')->nullable()->after('collection_datetime');
            $table->text('collection_details')->nullable()->after('collection_location');
            
            // Status field
            $table->enum('status', [
                'collected', 'in_storage', 'in_analysis', 'analyzed', 
                'in_court', 'returned', 'destroyed'
            ])->default('collected')->after('storage_location');
            
            // Condition field
            $table->enum('condition', [
                'excellent', 'good', 'fair', 'poor', 'damaged'
            ])->nullable()->after('status');
            
            // Current custodian
            $table->string('custodian')->nullable()->after('condition');
            
            // Notes field
            $table->text('notes')->nullable()->after('admissibility_notes');
            
            // Multiple photos and documents (JSON arrays of file paths)
            $table->json('photos')->nullable()->after('photo_path');
            $table->json('documents')->nullable()->after('photos');
            
            // Chain of custody as JSON
            $table->json('chain_of_custody')->nullable()->after('documents');
            
            // User tracking
            $table->foreignId('created_by_user_id')->nullable()->constrained('users')->after('notes');
            $table->foreignId('updated_by_user_id')->nullable()->constrained('users')->after('created_by_user_id');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::table('case_evidences', function (Blueprint $table) {
            // Revert added columns
            $table->dropColumn([
                'collection_time',
                'collection_details',
                'status',
                'condition',
                'custodian',
                'notes',
                'photos',
                'documents',
                'chain_of_custody'
            ]);
            
            // Revert renamed columns
            $table->renameColumn('tag_number', 'evidence_number');
            $table->renameColumn('evidence_type', 'type');
            $table->renameColumn('collection_datetime', 'collection_date');
            
            // Revert user tracking
            $table->dropConstrainedForeignId('created_by_user_id');
            $table->dropConstrainedForeignId('updated_by_user_id');
        });
        
        // Rename the table back to original name
        Schema::rename('case_evidences', 'case_evidence');
    }
};
