<?php

namespace Database\Seeders;

use App\Models\User;
use Carbon\Carbon;
use App\Models\CaseRecord;
use App\Models\CaseSuspect;
use App\Models\BailCondition;
use App\Models\BailApplication;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Database\Console\Seeds\WithoutModelEvents;

class BailApplicationSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Get existing data
        $cases = CaseRecord::all();
        $users = User::all();
        
        if ($cases->isEmpty()) {
            $this->command->error('No cases found in the database. Please seed cases first.');
            return;
        }
        
        // Clear existing bail data to avoid duplicates
        DB::statement('SET FOREIGN_KEY_CHECKS=0;');
        BailCondition::truncate();
        BailApplication::truncate();
        DB::statement('SET FOREIGN_KEY_CHECKS=1;');
        
        $this->command->info('Creating bail applications...');
        
        // Create bail applications for different scenarios
        foreach ($cases as $index => $case) {
            // Get suspects for this case
            $suspects = CaseSuspect::where('case_id', $case->id)->get();
            
            if ($suspects->isEmpty()) {
                continue; // Skip cases with no suspects
            }
            
            // Create 1-3 bail applications per case
            $bailCount = rand(1, min(3, $suspects->count()));
            
            for ($i = 0; $i < $bailCount; $i++) {
                // Select a random suspect from this case
                $suspect = $suspects->random();
                
                // Generate application date (between case created date and now)
                $caseDate = $case->created_at ?? Carbon::now()->subMonths(6);
                $applicationDate = Carbon::parse($caseDate)->addDays(rand(1, 30));
                
                // Generate a unique application number
                $applicationNumber = 'BAIL-' . date('Y') . '-' . str_pad(rand(1, 999), 3, '0', STR_PAD_LEFT);
                
                // Randomly select a status
                $status = ['pending', 'approved', 'rejected', 'revoked'][rand(0, 3)];
                
                // Create the bail application
                $bailApplication = BailApplication::create([
                    'case_id' => $case->id,
                    'suspect_id' => $suspect->id,
                    'application_number' => $applicationNumber,
                    'application_date' => $applicationDate,
                    'status' => $status,
                    'reason' => 'The suspect has requested bail on the grounds of ' . [
                        'being the primary caregiver for dependents',
                        'medical reasons requiring specialized treatment',
                        'having strong community ties and no flight risk',
                        'being a first-time offender with no prior criminal record',
                        'cooperating fully with the investigation'
                    ][rand(0, 4)],
                    'notes' => rand(0, 1) ? 'Additional investigation ongoing. Regular updates required.' : null,
                    'created_by' => $users->random()->id,
                    'updated_by' => $users->random()->id,
                ]);
                
                // For approved applications, add approval details
                if ($status === 'approved') {
                    $approvalDate = Carbon::parse($applicationDate)->addDays(rand(1, 14));
                    $expiryDate = Carbon::parse($approvalDate)->addMonths(rand(1, 6));
                    
                    $bailApplication->update([
                        'approval_date' => $approvalDate,
                        'expiry_date' => $expiryDate,
                        'amount' => rand(1, 10) * 5000, // Random amount between 5,000 and 50,000
                        'approved_by' => $users->random()->id,
                        'surety_name' => ['John Doe', 'Jane Smith', 'Mohammed Ali', 'Fatou Ceesay', 'Omar Jallow'][rand(0, 4)],
                        'surety_contact' => '+220' . rand(7000000, 9999999),
                        'surety_address' => ['Banjul', 'Serrekunda', 'Bakau', 'Brikama', 'Gunjur'][rand(0, 4)] . ' Area, The Gambia',
                        'surety_relationship' => ['Family Member', 'Friend', 'Colleague', 'Employer', 'Community Leader'][rand(0, 4)],
                        'surety_id_type' => ['National ID', 'Passport', 'Driver License', 'Voter Card'][rand(0, 3)],
                        'surety_id_number' => strtoupper(substr(md5(rand()), 0, 8)),
                    ]);
                    
                    // Add bail conditions (2-5 conditions)
                    $conditionCount = rand(2, 5);
                    $conditionTypes = [
                        'reporting', 'travel', 'residence', 'contact', 
                        'curfew', 'passport', 'employment', 'treatment', 'other'
                    ];
                    
                    for ($j = 0; $j < $conditionCount; $j++) {
                        $conditionType = $conditionTypes[array_rand($conditionTypes)];
                        $isFulfilled = (rand(0, 1) == 1);
                        
                        $condition = [
                            'bail_application_id' => $bailApplication->id,
                            'condition_type' => $conditionType,
                            'description' => $this->getConditionDescription($conditionType),
                            'is_fulfilled' => $isFulfilled,
                            'created_by' => $users->random()->id,
                        ];
                        
                        if ($isFulfilled) {
                            $fulfillmentDate = Carbon::parse($approvalDate)->addDays(rand(7, 60));
                            $condition['fulfillment_date'] = $fulfillmentDate;
                            $condition['fulfillment_notes'] = 'Verified and confirmed compliance with condition.';
                            $condition['verified_by'] = $users->random()->id;
                        }
                        
                        BailCondition::create($condition);
                    }
                }
            }
        }
        
        $this->command->info(BailApplication::count() . ' bail applications created with ' . BailCondition::count() . ' conditions.');
    }
    
    /**
     * Get a realistic description for a bail condition based on its type.
     *
     * @param string $conditionType
     * @return string
     */
    private function getConditionDescription(string $conditionType): string
    {
        return match($conditionType) {
            'reporting' => 'Must report to ' . ['Banjul', 'Serrekunda', 'Bakau', 'Brikama'][rand(0, 3)] . ' Police Station every ' . ['Monday', 'Wednesday', 'Friday', 'Monday and Thursday'][rand(0, 3)] . ' between 9am and 12pm.',
            'travel' => 'Must not leave The Gambia and must surrender all travel documents to the court.',
            'residence' => 'Must reside at the address provided to the court and notify of any change of address.',
            'contact' => 'Must not contact, directly or indirectly, any prosecution witnesses or complainants in this case.',
            'curfew' => 'Must remain at place of residence between ' . ['8pm', '9pm', '10pm'][rand(0, 2)] . ' and 6am daily.',
            'passport' => 'Must surrender passport and all travel documents to the court within 24 hours.',
            'employment' => 'Must maintain employment or actively seek employment and provide proof to the court.',
            'treatment' => 'Must attend all scheduled ' . ['medical', 'psychological', 'substance abuse'][rand(0, 2)] . ' treatment sessions.',
            'other' => ['Must not enter specified exclusion zones.', 'Must not possess any weapons.', 'Must abstain from alcohol and drugs.'][rand(0, 2)],
            default => 'Must comply with all court orders and attend all scheduled court appearances.',
        };
    }
}
