<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\BailApplication;
use App\Models\BailCondition;
use App\Models\BailCheckin;
use App\Models\CaseSuspect;
use App\Models\PoliceStation;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class BailCheckinSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Get existing data
        $bailApplications = BailApplication::where('status', 'approved')->get();
        $policeStations = PoliceStation::all();
        $users = User::all();
        
        if ($bailApplications->isEmpty()) {
            $this->command->error('No approved bail applications found. Cannot create check-ins.');
            return;
        }
        
        // Clear existing check-ins
        DB::table('bail_checkins')->truncate();
        
        $this->command->info('Creating bail check-ins...');
        
        // Create check-ins for each bail application with reporting conditions
        foreach ($bailApplications as $application) {
            // Get reporting conditions for this application
            $reportingConditions = BailCondition::where('bail_application_id', $application->id)
                ->where('requires_reporting', true)
                ->get();
                
            if ($reportingConditions->isEmpty()) {
                continue;
            }
            
            $suspect = $application->suspect;
            if (!$suspect) {
                continue;
            }
            
            foreach ($reportingConditions as $condition) {
                $policeStation = $condition->police_station_id 
                    ? PoliceStation::find($condition->police_station_id) 
                    : $policeStations->random();
                
                // Create past check-ins (mix of present, missed, excused)
                $this->createPastCheckIns($application, $suspect, $condition, $policeStation, $users);
                
                // Create today's check-in
                $this->createTodayCheckIn($application, $suspect, $condition, $policeStation, $users);
                
                // Create future scheduled check-ins
                $this->createFutureCheckIns($application, $suspect, $condition, $policeStation, $users);
            }
        }
        
        $this->command->info('Created ' . BailCheckin::count() . ' bail check-ins successfully!');
    }
    
    /**
     * Create past check-ins with various statuses
     */
    private function createPastCheckIns($application, $suspect, $condition, $policeStation, $users)
    {
        // Create check-ins for the past 90 days based on reporting frequency
        // This gives us more data for the statistics view
        $startDate = Carbon::now()->subDays(90);
        $endDate = Carbon::yesterday();
        
        $dates = $this->getCheckInDates($condition, $startDate, $endDate);
        
        foreach ($dates as $date) {
            // Randomize status (80% present, 15% missed, 5% excused)
            $rand = rand(1, 100);
            if ($rand <= 80) {
                $status = 'present';
                $isPresent = true;
            } elseif ($rand <= 95) {
                $status = 'missed';
                $isPresent = false;
            } else {
                $status = 'excused';
                $isPresent = false;
            }
            
            // Create the check-in
            BailCheckin::create([
                'bail_application_id' => $application->id,
                'suspect_id' => $suspect->id,
                'police_station_id' => $policeStation->id,
                'checkin_time' => $date->copy()->addHours(rand(9, 16))->addMinutes(rand(0, 59)),
                'scheduled_time' => $date->copy()->setHour(10)->setMinute(0),
                'is_present' => $isPresent,
                'status' => $status,
                'notes' => $status == 'excused' ? $this->getRandomExcuseNote() : null,
                'recorded_by' => $users->random()->id,
                'created_by' => $users->random()->id,
                'updated_by' => $users->random()->id,
                'created_at' => $date,
                'updated_at' => $date,
            ]);
        }
    }
    
    /**
     * Create today's check-in with various statuses
     */
    private function createTodayCheckIn($application, $suspect, $condition, $policeStation, $users)
    {
        // Check if today is a reporting day
        $today = Carbon::today();
        if (!$this->isReportingDay($condition, $today)) {
            return;
        }
        
        // Randomize status for today (40% present, 30% missed, 20% scheduled, 10% excused)
        $rand = rand(1, 100);
        $scheduledTime = $today->copy()->setHour(10)->setMinute(0);
        
        if ($rand <= 40) {
            $status = 'present';
            $isPresent = true;
            $checkinTime = $today->copy()->addHours(rand(9, 12))->addMinutes(rand(0, 59));
        } elseif ($rand <= 70) {
            $status = 'missed';
            $isPresent = false;
            $checkinTime = $scheduledTime; // Use scheduled time instead of null
        } elseif ($rand <= 90) {
            $status = 'scheduled';
            $isPresent = false;
            $checkinTime = $scheduledTime; // Use scheduled time instead of null
        } else {
            $status = 'excused';
            $isPresent = false;
            $checkinTime = $scheduledTime; // Use scheduled time instead of null
        }
        
        // Create the check-in
        BailCheckin::create([
            'bail_application_id' => $application->id,
            'suspect_id' => $suspect->id,
            'police_station_id' => $policeStation->id,
            'checkin_time' => $checkinTime,
            'scheduled_time' => $scheduledTime,
            'is_present' => $isPresent,
            'status' => $status,
            'notes' => $status == 'excused' ? $this->getRandomExcuseNote() : null,
            'recorded_by' => $users->random()->id,
            'created_by' => $users->random()->id,
            'updated_by' => $users->random()->id,
            'created_at' => $today,
            'updated_at' => $today,
        ]);
    }
    
    /**
     * Create future scheduled check-ins
     */
    private function createFutureCheckIns($application, $suspect, $condition, $policeStation, $users)
    {
        // Create check-ins for the next 14 days based on reporting frequency
        $startDate = Carbon::tomorrow();
        $endDate = Carbon::now()->addDays(14);
        
        $dates = $this->getCheckInDates($condition, $startDate, $endDate);
        
        foreach ($dates as $date) {
            // All future check-ins are scheduled
            // For future check-ins, set checkin_time to the same as scheduled_time
            // but we'll mark it as scheduled status and not present
            $scheduledTime = $date->copy()->setHour(10)->setMinute(0);
            
            BailCheckin::create([
                'bail_application_id' => $application->id,
                'suspect_id' => $suspect->id,
                'police_station_id' => $policeStation->id,
                'checkin_time' => $scheduledTime, // Use scheduled time instead of null
                'scheduled_time' => $scheduledTime,
                'is_present' => false,
                'status' => 'scheduled',
                'notes' => null,
                'recorded_by' => $users->random()->id,
                'created_by' => $users->random()->id,
                'updated_by' => $users->random()->id,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]);
        }
    }
    
    /**
     * Get check-in dates based on reporting frequency
     */
    private function getCheckInDates($condition, $startDate, $endDate)
    {
        $dates = [];
        $currentDate = $startDate->copy();
        
        while ($currentDate->lte($endDate)) {
            if ($this->isReportingDay($condition, $currentDate)) {
                $dates[] = $currentDate->copy();
            }
            $currentDate->addDay();
        }
        
        return $dates;
    }
    
    /**
     * Check if the given date is a reporting day based on condition
     */
    private function isReportingDay($condition, $date)
    {
        switch ($condition->reporting_frequency) {
            case 'daily':
                return true;
                
            case 'weekly':
                $reportingDays = $condition->reporting_days ? json_decode($condition->reporting_days) : ['Monday'];
                return in_array($date->format('l'), $reportingDays);
                
            case 'biweekly':
                $reportingDays = $condition->reporting_days ? json_decode($condition->reporting_days) : ['Monday'];
                $weekNumber = $date->weekOfYear;
                return in_array($date->format('l'), $reportingDays) && $weekNumber % 2 == 0;
                
            case 'monthly':
                $reportingDays = $condition->reporting_days ? json_decode($condition->reporting_days) : [1];
                return in_array($date->day, $reportingDays);
                
            case 'custom':
                // For custom, we'll just use Mondays, Wednesdays and Fridays
                return in_array($date->format('l'), ['Monday', 'Wednesday', 'Friday']);
                
            default:
                return false;
        }
    }
    
    /**
     * Get a random excuse note
     */
    private function getRandomExcuseNote()
    {
        $excuses = [
            'Medical emergency - hospitalized with documentation provided',
            'Vehicle breakdown on the way to station - towing receipt provided',
            'Death in immediate family - funeral documentation provided',
            'Severe weather conditions prevented travel - confirmed by officer',
            'Court appearance for unrelated matter - court summons provided',
            'Child care emergency - verified by social worker',
            'Employer required mandatory overtime - supervisor confirmation provided',
            'Public transportation strike affected travel - news report confirmed',
        ];
        
        return $excuses[array_rand($excuses)];
    }
}
