<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\BailApplication;
use App\Models\BailCondition;
use App\Models\PoliceStation;
use App\Models\User;
use Carbon\Carbon;

class BailConditionSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Get existing data
        $bailApplications = BailApplication::where('status', 'approved')->get();
        $policeStations = PoliceStation::all();
        $users = User::all();
        
        if ($bailApplications->isEmpty()) {
            $this->command->error('No approved bail applications found. Cannot create conditions.');
            return;
        }
        
        // Clear existing conditions
        BailCondition::truncate();
        
        $this->command->info('Creating bail conditions...');
        
        // Common condition types
        $conditionTypes = [
            'Travel Restriction',
            'Reporting Requirement',
            'Curfew',
            'No Contact Order',
            'Substance Testing',
            'Employment Requirement',
            'Surrender of Documents',
            'Electronic Monitoring'
        ];
        
        // Reporting frequencies
        $frequencies = ['daily', 'weekly', 'biweekly', 'monthly', 'custom'];
        
        // Create conditions for each bail application
        foreach ($bailApplications as $application) {
            // Each application gets 2-4 conditions
            $numConditions = rand(2, 4);
            $selectedTypes = array_rand(array_flip($conditionTypes), $numConditions);
            
            if (!is_array($selectedTypes)) {
                $selectedTypes = [$selectedTypes];
            }
            
            // Ensure 'Reporting Requirement' is always included
            if (!in_array('Reporting Requirement', $selectedTypes)) {
                $selectedTypes[] = 'Reporting Requirement';
            }
            
            foreach ($selectedTypes as $type) {
                $requiresReporting = $type === 'Reporting Requirement';
                $policeStation = $policeStations->random();
                
                // Create the condition
                $condition = new BailCondition();
                $condition->bail_application_id = $application->id;
                $condition->condition_type = $type;
                $condition->description = $this->getConditionDescription($type);
                $condition->requires_reporting = $requiresReporting;
                
                // If reporting is required, set up the schedule
                if ($requiresReporting) {
                    $frequency = $frequencies[array_rand($frequencies)];
                    $condition->police_station_id = $policeStation->id;
                    $condition->reporting_frequency = $frequency;
                    
                    // Set reporting days based on frequency
                    switch ($frequency) {
                        case 'daily':
                            $condition->reporting_days = null; // Every day
                            break;
                        case 'weekly':
                            $days = $this->getRandomWeekdays(1);
                            $condition->reporting_days = json_encode($days);
                            break;
                        case 'biweekly':
                            $days = $this->getRandomWeekdays(1);
                            $condition->reporting_days = json_encode($days);
                            break;
                        case 'monthly':
                            $days = [rand(1, 28)]; // Random day of month
                            $condition->reporting_days = json_encode($days);
                            break;
                        case 'custom':
                            $days = $this->getRandomWeekdays(rand(2, 3));
                            $condition->reporting_days = json_encode($days);
                            break;
                    }
                    
                    // Set reporting time window
                    $startHour = rand(8, 10);
                    $endHour = rand(15, 17);
                    $condition->reporting_time_start = sprintf('%02d:00:00', $startHour);
                    $condition->reporting_time_end = sprintf('%02d:00:00', $endHour);
                    
                    // Set reporting date range
                    $startDate = Carbon::parse($application->approval_date);
                    $endDate = $application->expiry_date 
                        ? Carbon::parse($application->expiry_date) 
                        : $startDate->copy()->addMonths(rand(3, 6));
                    
                    $condition->reporting_start_date = $startDate->format('Y-m-d');
                    $condition->reporting_end_date = $endDate->format('Y-m-d');
                    
                    // Calculate total required check-ins
                    $totalDays = $startDate->diffInDays($endDate);
                    $totalCheckins = $this->calculateTotalCheckins($frequency, $totalDays, $condition->reporting_days);
                    $condition->total_required_checkins = $totalCheckins;
                    
                    // Set random number of completed check-ins
                    $condition->completed_checkins = rand(0, min(10, $totalCheckins));
                }
                
                // Set user tracking
                $condition->created_by = $users->random()->id;
                $condition->updated_by = $users->random()->id;
                
                $condition->save();
            }
        }
        
        $this->command->info('Created ' . BailCondition::count() . ' bail conditions successfully!');
    }
    
    /**
     * Get a detailed description for a condition type
     */
    private function getConditionDescription($type)
    {
        $descriptions = [
            'Travel Restriction' => 'Must not leave the jurisdiction of Gambia without prior court approval. Must surrender passport to court registry.',
            'Reporting Requirement' => 'Must report to the designated police station as scheduled. Failure to report may result in revocation of bail.',
            'Curfew' => 'Must remain at registered address between 8:00 PM and 6:00 AM daily unless authorized by supervising officer.',
            'No Contact Order' => 'Must not contact or communicate with the victim or witnesses directly or indirectly through any means.',
            'Substance Testing' => 'Must submit to random drug and alcohol testing when requested by supervising officer.',
            'Employment Requirement' => 'Must maintain or actively seek employment and provide proof to supervising officer.',
            'Surrender of Documents' => 'Must surrender all travel documents including passport and national ID to court registry within 24 hours.',
            'Electronic Monitoring' => 'Must wear electronic monitoring device at all times and maintain the equipment in working order.'
        ];
        
        return $descriptions[$type] ?? 'Condition details to be determined by supervising officer.';
    }
    
    /**
     * Get random weekdays
     */
    private function getRandomWeekdays($count)
    {
        $weekdays = ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday'];
        shuffle($weekdays);
        return array_slice($weekdays, 0, $count);
    }
    
    /**
     * Calculate total check-ins based on frequency and date range
     */
    private function calculateTotalCheckins($frequency, $totalDays, $reportingDays)
    {
        switch ($frequency) {
            case 'daily':
                return $totalDays;
                
            case 'weekly':
                return intval(ceil($totalDays / 7));
                
            case 'biweekly':
                return intval(ceil($totalDays / 14));
                
            case 'monthly':
                return intval(ceil($totalDays / 30));
                
            case 'custom':
                $days = json_decode($reportingDays);
                $daysPerWeek = count($days);
                return intval(ceil(($totalDays / 7) * $daysPerWeek));
                
            default:
                return 0;
        }
    }
}
