<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use App\Models\CaseEvidence;
use App\Models\CaseRecord;
use App\Models\User;
use Carbon\Carbon;

class CaseEvidenceSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $cases = CaseRecord::all();
        $officers = User::where('is_officer', true)->get();
        
        $evidences = [
            [
                'tag_number' => 'EV-2023-001',
                'evidence_type' => 'physical',
                'subtype' => 'weapon',
                'description' => 'Replica firearm used in robbery. Black in color, resembling a Glock 19.',
                'collection_date' => Carbon::now()->subDays(5)->format('Y-m-d'),
                'collection_time' => '14:30:00',
                'collection_location' => 'Kairaba Avenue, Serrekunda',
                'collection_method' => 'Manual recovery',
                'condition' => 'good',
                'status' => 'collected',
                'notes' => 'Found discarded in trash bin 100 meters from crime scene. No fingerprints recovered.',
            ],
            [
                'tag_number' => 'EV-2023-002',
                'evidence_type' => 'documentary',
                'subtype' => 'vehicle_document',
                'description' => 'Vehicle registration documents for stolen Toyota Corolla, registration number BJL 1234.',
                'collection_date' => Carbon::now()->subDays(3)->format('Y-m-d'),
                'collection_time' => '10:15:00',
                'collection_location' => 'Victim\'s residence, Kololi',
                'collection_method' => 'Provided by victim',
                'condition' => 'good',
                'status' => 'collected',
                'notes' => 'Original vehicle documents provided by the victim for verification purposes.',
            ],
            [
                'tag_number' => 'EV-2023-003',
                'evidence_type' => 'forensic',
                'subtype' => 'blood_sample',
                'description' => 'Blood sample collected from victim of assault at Westfield Junction.',
                'collection_date' => Carbon::now()->subDays(2)->format('Y-m-d'),
                'collection_time' => '16:45:00',
                'collection_location' => 'Serrekunda General Hospital',
                'collection_method' => 'Medical collection',
                'condition' => 'good',
                'status' => 'in_analysis',
                'notes' => 'Sample collected by medical staff and handed over to investigating officer.',
            ],
            [
                'tag_number' => 'EV-2023-004',
                'evidence_type' => 'digital',
                'subtype' => 'video',
                'description' => 'CCTV footage from Fajara Electronics Store showing burglary in progress.',
                'collection_date' => Carbon::now()->subDays(6)->format('Y-m-d'),
                'collection_time' => '09:30:00',
                'collection_location' => 'Fajara Electronics Store, Fajara',
                'collection_method' => 'Digital copy',
                'condition' => 'good',
                'status' => 'analyzed',
                'notes' => 'Footage shows two suspects breaking into the store at approximately 02:15 AM.',
            ],
            [
                'tag_number' => 'EV-2023-005',
                'evidence_type' => 'physical',
                'subtype' => 'controlled_substance',
                'description' => 'Suspected cannabis, approximately 50 grams, found in suspect\'s possession.',
                'collection_date' => Carbon::now()->subDay()->format('Y-m-d'),
                'collection_time' => '11:20:00',
                'collection_location' => 'Banjul Ferry Terminal',
                'collection_method' => 'Seized during search',
                'condition' => 'good',
                'status' => 'in_storage',
                'notes' => 'Substance to be sent to lab for testing and confirmation.',
            ],
        ];
        
        foreach ($evidences as $index => $evidence) {
            $caseIndex = $index % count($cases->toArray());
            $officerIndex = $index % count($officers->toArray());
            
            // Create a chain of custody entry for the evidence
            $chainOfCustody = [
                [
                    'date' => $evidence['collection_date'],
                    'time' => $evidence['collection_time'],
                    'action' => 'collected',
                    'handler' => $officers[$officerIndex]->first_name . ' ' . $officers[$officerIndex]->last_name,
                    'badge_number' => $officers[$officerIndex]->badge_number,
                    'location' => $evidence['collection_location'],
                    'notes' => 'Initial collection of evidence',
                ],
                [
                    'date' => Carbon::parse($evidence['collection_date'])->addHours(2)->format('Y-m-d'),
                    'time' => Carbon::parse($evidence['collection_time'])->addHours(2)->format('H:i:s'),
                    'action' => 'stored',
                    'handler' => $officers[$officerIndex]->first_name . ' ' . $officers[$officerIndex]->last_name,
                    'badge_number' => $officers[$officerIndex]->badge_number,
                    'location' => 'Evidence Room, ' . $cases[$caseIndex]->station->name,
                    'notes' => 'Evidence logged and stored in secure evidence room',
                ],
            ];
            
            // Check if evidence already exists to avoid duplicates
            $existingEvidence = CaseEvidence::where('tag_number', $evidence['tag_number'])->first();
            
            if (!$existingEvidence) {
                $collectionDateTime = Carbon::parse($evidence['collection_date'] . ' ' . $evidence['collection_time']);
                
                CaseEvidence::create([
                    'case_id' => $cases[$caseIndex]->id,
                    'tag_number' => $evidence['tag_number'],
                    'name' => 'Evidence ' . $evidence['tag_number'],
                    'evidence_type' => $evidence['evidence_type'],
                    'subtype' => $evidence['subtype'],
                    'description' => $evidence['description'],
                    'collection_datetime' => $collectionDateTime,
                    'collection_time' => $collectionDateTime->format('H:i:s'),
                    'collection_location' => $evidence['collection_location'],
                    'collection_details' => $evidence['collection_method'],
                    'status' => $evidence['status'],
                    'condition' => $evidence['condition'],
                    'chain_of_custody_status' => 'in_storage',
                    'chain_of_custody' => json_encode($chainOfCustody),
                    'collected_by_officer_id' => $officers[$officerIndex]->id,
                    'storage_location' => 'Evidence Room, ' . $cases[$caseIndex]->station->name,
                    'is_sensitive' => false,
                    'is_admissible' => true,
                    'notes' => $evidence['notes'],
                    'photos' => json_encode([]),
                    'documents' => json_encode([]),
                    'created_by_user_id' => $officers[$officerIndex]->id,
                    'updated_by_user_id' => $officers[$officerIndex]->id,
                ]);
            }
        }
    }
}
