<?php

namespace Database\Seeders;

use App\Models\CaseRecord;
use App\Models\CaseStatement;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Faker\Factory as Faker;

class CaseStatementSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $faker = Faker::create();
        
        // Get all cases
        $cases = CaseRecord::all();
        
        // Get officers for recording statements
        $officers = User::where('is_officer', true)->get();
        
        if ($cases->isEmpty() || $officers->isEmpty()) {
            $this->command->info('No cases or officers found. Please run CaseRecordSeeder and UserSeeder first.');
            return;
        }
        
        // Statement types
        $statementTypes = ['victim', 'witness', 'suspect', 'officer', 'expert'];
        
        // ID types
        $idTypes = ['national_id', 'passport', 'drivers_license', 'voter_id', 'other'];
        
        // Check if we already have statements
        if (DB::table('case_statements')->count() > 0) {
            $this->command->info('Case statements table already has data. Skipping seeding.');
            return;
        }
        
        $this->command->info('Seeding case statements...');
        
        // Create 3-5 statements for each case
        foreach ($cases as $case) {
            $statementsCount = rand(3, 5);
            
            for ($i = 0; $i < $statementsCount; $i++) {
                $statementType = $faker->randomElement($statementTypes);
                $recordingOfficer = $faker->randomElement($officers);
                $statementDate = Carbon::parse($case->incident_date)->addDays(rand(1, 14));
                
                // For some statements, add a verifying officer
                $isVerified = $faker->boolean(70); // 70% chance of being verified
                $verifiedByOfficerId = null;
                $verifiedAt = null;
                
                if ($isVerified) {
                    $verifiedByOfficerId = $faker->randomElement($officers)->id;
                    $verifiedAt = $statementDate->copy()->addDays(rand(1, 3));
                }
                
                // Generate appropriate name based on statement type
                $givenByName = '';
                switch ($statementType) {
                    case 'victim':
                        // Try to use an actual victim name if available
                        $victim = $case->victims()->first();
                        $givenByName = $victim ? ($victim->full_name ?? $victim->name) : $faker->name;
                        break;
                    case 'suspect':
                        // Try to use an actual suspect name if available
                        $suspect = $case->suspects()->first();
                        $givenByName = $suspect ? $suspect->full_name : $faker->name;
                        break;
                    case 'officer':
                        // Use an officer name
                        $givenByName = $faker->randomElement($officers)->name;
                        break;
                    default:
                        // For witnesses and experts, use random names
                        $givenByName = $faker->name;
                }
                
                // Generate statement content based on type
                $statementContent = '';
                switch ($statementType) {
                    case 'victim':
                        $statementContent = "I, {$givenByName}, was the victim of " . strtolower($case->crime_type) . " on " . $case->incident_date->format('F j, Y') . " at approximately " . $faker->time('h:i A') . ". " . 
                            $faker->paragraph(3) . "\n\n" . 
                            "The incident occurred at " . $case->incident_location . ". " . $faker->paragraph(2) . "\n\n" . 
                            "I can identify the perpetrator as " . $faker->optional(0.7, "an unknown person")->passthrough($faker->name) . ". " . $faker->paragraph(2);
                        break;
                    case 'witness':
                        $statementContent = "I, {$givenByName}, witnessed an incident on " . $case->incident_date->format('F j, Y') . " at approximately " . $faker->time('h:i A') . ". " . 
                            $faker->paragraph(2) . "\n\n" . 
                            "I was at " . $case->incident_location . " when I observed " . $faker->sentence(10) . ". " . $faker->paragraph(3) . "\n\n" . 
                            "I can describe the suspect as " . $faker->sentence(8) . ".";
                        break;
                    case 'suspect':
                        $statementContent = "I, {$givenByName}, provide this statement regarding the allegations against me. " . 
                            $faker->paragraph(2) . "\n\n" . 
                            "On " . $case->incident_date->format('F j, Y') . ", I was " . $faker->sentence(10) . ". " . $faker->paragraph(3) . "\n\n" . 
                            "I deny any involvement in the alleged " . strtolower($case->crime_type) . " and wish to state that " . $faker->paragraph(1);
                        break;
                    case 'officer':
                        $statementContent = "I, Officer {$givenByName}, badge number " . $faker->numberBetween(1000, 9999) . ", was on duty on " . $case->incident_date->format('F j, Y') . ". " . 
                            "I was dispatched to " . $case->incident_location . " at approximately " . $faker->time('h:i A') . " following a report of " . strtolower($case->crime_type) . ". " . 
                            $faker->paragraph(3) . "\n\n" . 
                            "Upon arrival, I observed " . $faker->paragraph(2) . "\n\n" . 
                            "I proceeded to " . $faker->sentence(8) . ". " . $faker->paragraph(2);
                        break;
                    case 'expert':
                        $statementContent = "I, {$givenByName}, a qualified expert in " . $faker->jobTitle() . ", provide this professional assessment regarding case #" . $case->case_number . ". " . 
                            $faker->paragraph(2) . "\n\n" . 
                            "Based on my examination of the evidence, I conclude that " . $faker->paragraph(3) . "\n\n" . 
                            "My professional opinion is that " . $faker->paragraph(2) . "\n\n" . 
                            "This assessment is based on my " . $faker->numberBetween(5, 30) . " years of experience in the field.";
                        break;
                }
                
                // Create the statement
                CaseStatement::create([
                    'case_id' => $case->id,
                    'statement_type' => $statementType,
                    'given_by_name' => $givenByName,
                    'given_by_contact' => $faker->phoneNumber,
                    'given_by_address' => $faker->address,
                    'given_by_id_type' => $faker->randomElement($idTypes),
                    'given_by_id_number' => $faker->unique()->numerify('ID-#######'),
                    'recorded_by_officer_id' => $recordingOfficer->id,
                    'statement_date' => $statementDate,
                    'statement_content' => $statementContent,
                    'location_taken' => 'Police Station - ' . $faker->city,
                    'is_sworn' => $faker->boolean(80),
                    'is_signed' => $faker->boolean(90),
                    'officer_remarks' => $faker->optional(0.7)->sentence(10),
                    'is_verified' => $isVerified,
                    'verified_by_officer_id' => $verifiedByOfficerId,
                    'verified_at' => $verifiedAt,
                ]);
            }
        }
        
        $this->command->info('Case statements seeded successfully.');
    }
}
