<?php

namespace Database\Seeders;

use App\Models\Evidence;
use App\Models\User;
use App\Models\CaseRecord;
use App\Models\PoliceStation;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Carbon\Carbon;

class EvidenceSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Create a test police station if none exists
        $station = PoliceStation::firstOrCreate(
            ['code' => 'BAN-HQ'],
            [
                'name' => 'Banjul Police Headquarters',
                'code' => 'BAN-HQ',
                'address' => '123 Police Plaza',
                'city' => 'Banjul',
                'region' => 'Greater Banjul',
                'phone' => '123-456-7890',
                'email' => 'headquarters@police.gov.gm',
                'description' => 'Main police headquarters for Banjul',
                'is_headquarters' => true,
                'is_active' => true,
            ]
        );
        
        // Create a test user if none exists
        $user = User::firstOrCreate(
            ['email' => 'officer@example.com'],
            [
                'name' => 'Test Officer',
                'email' => 'officer@example.com',
                'password' => Hash::make('password'),
                'is_officer' => true,
                'rank' => 'Inspector',
                'badge_number' => 'B12345',
                'station_id' => $station->id,
            ]
        );
        
        // Create a test case if none exists
        $case = CaseRecord::firstOrCreate(
            ['case_number' => 'CASE-2025-001'],
            [
                'case_number' => 'CASE-2025-001',
                'title' => 'Test Robbery Case',
                'description' => 'A test case for evidence management',
                'status' => 'open',
                'priority' => 'high',
                'crime_type' => 'robbery',
                'crime_category' => 'violent',
                'incident_date' => Carbon::now()->subDays(5),
                'reported_date' => Carbon::now()->subDays(4),
                'incident_location' => 'Banjul Central',
                'created_by_user_id' => $user->id,
                'updated_by_user_id' => $user->id,
                'station_id' => $station->id,
                'lead_investigator_id' => $user->id,
                'reporting_officer_id' => $user->id,
            ]
        );
        
        // Create sample evidence items
        $evidenceItems = [
            [
                'tag_number' => 'EV-2025-001',
                'evidence_type' => 'weapon',
                'description' => 'Kitchen knife with 8-inch blade',
                'status' => 'in_custody',
                'condition' => 'Good condition with fingerprints preserved',
                'location' => 'Evidence Room A',
                'storage_location' => 'Locker 12, Shelf B',
                'collection_date' => Carbon::now()->subDays(4),
                'collection_location' => 'Victim\'s kitchen, 123 Main St, Banjul',
                'collection_notes' => 'Found on kitchen counter with blood residue',
                'collected_by_officer_id' => $user->id,
                'custodian' => $user->id,
                'photos' => json_encode([
                    [], // First element is empty array
                    [
                        'path' => 'evidence/photos/sample-knife.jpg',
                        'name' => 'Kitchen Knife - Front View',
                        'uploaded_at' => Carbon::now()->subDays(4)->format('Y-m-d H:i:s'),
                        'uploaded_by' => $user->id,
                    ],
                    [
                        'path' => 'evidence/photos/sample-knife-handle.jpg',
                        'name' => 'Kitchen Knife - Handle Detail',
                        'uploaded_at' => Carbon::now()->subDays(4)->format('Y-m-d H:i:s'),
                        'uploaded_by' => $user->id,
                    ],
                ]),
                'documents' => json_encode([
                    [], // First element is empty array
                    [
                        'path' => 'evidence/documents/knife-report.pdf',
                        'name' => 'Initial Assessment Report',
                        'uploaded_at' => Carbon::now()->subDays(3)->format('Y-m-d H:i:s'),
                        'uploaded_by' => $user->id,
                    ],
                ]),
                'chain_of_custody' => json_encode([
                    [
                        'date' => Carbon::now()->subDays(4)->format('Y-m-d H:i:s'),
                        'from_officer_id' => null,
                        'to_officer_id' => $user->id,
                        'action' => 'collected',
                        'notes' => 'Initial collection from crime scene',
                    ]
                ]),
                'case_id' => $case->id,
                'created_by_user_id' => $user->id,
                'updated_by_user_id' => $user->id,
            ],
            [
                'tag_number' => 'EV-2025-002',
                'evidence_type' => 'physical',
                'description' => 'Torn shirt with blood stains',
                'status' => 'in_analysis',
                'condition' => 'Torn fabric with visible blood stains',
                'location' => 'Forensic Lab',
                'storage_location' => 'Lab 3, Cabinet 2',
                'collection_date' => Carbon::now()->subDays(4),
                'collection_location' => 'Victim\'s bedroom, 123 Main St, Banjul',
                'collection_notes' => 'Found on bedroom floor',
                'collected_by_officer_id' => $user->id,
                'custodian' => $user->id,
                'analyzed_by_officer_id' => $user->id,
                'analysis_date' => Carbon::now()->subDays(2),
                'analysis_results' => 'Preliminary tests confirm blood type matches victim',
                'photos' => json_encode([
                    [], // First element is empty array
                    [
                        'path' => 'evidence/photos/sample-shirt.jpg',
                        'name' => 'Torn Shirt - Full View',
                        'uploaded_at' => Carbon::now()->subDays(4)->format('Y-m-d H:i:s'),
                        'uploaded_by' => $user->id,
                    ],
                    [
                        'path' => 'evidence/photos/sample-shirt-stain.jpg',
                        'name' => 'Torn Shirt - Blood Stain Detail',
                        'uploaded_at' => Carbon::now()->subDays(4)->format('Y-m-d H:i:s'),
                        'uploaded_by' => $user->id,
                    ],
                ]),
                'documents' => json_encode([
                    [], // First element is empty array
                    [
                        'path' => 'evidence/documents/blood-analysis.pdf',
                        'name' => 'Blood Analysis Report',
                        'uploaded_at' => Carbon::now()->subDays(2)->format('Y-m-d H:i:s'),
                        'uploaded_by' => $user->id,
                    ],
                ]),
                'chain_of_custody' => json_encode([
                    [
                        'date' => Carbon::now()->subDays(4)->format('Y-m-d H:i:s'),
                        'from_officer_id' => null,
                        'to_officer_id' => $user->id,
                        'action' => 'collected',
                        'notes' => 'Initial collection from crime scene',
                    ],
                    [
                        'date' => Carbon::now()->subDays(3)->format('Y-m-d H:i:s'),
                        'from_officer_id' => $user->id,
                        'to_officer_id' => $user->id,
                        'action' => 'transferred',
                        'notes' => 'Transferred to forensic lab for analysis',
                    ]
                ]),
                'case_id' => $case->id,
                'created_by_user_id' => $user->id,
                'updated_by_user_id' => $user->id,
            ],
            [
                'tag_number' => 'EV-2025-003',
                'evidence_type' => 'digital',
                'description' => 'Security camera footage',
                'status' => 'in_custody',
                'condition' => 'Digital file, 2 hours of footage',
                'location' => 'Digital Evidence Server',
                'storage_location' => 'Server 1, Case Folder 2025-001',
                'collection_date' => Carbon::now()->subDays(3),
                'collection_location' => 'Security office at 125 Main St, Banjul',
                'collection_notes' => 'Retrieved with owner\'s permission',
                'collected_by_officer_id' => $user->id,
                'custodian' => $user->id,
                'photos' => json_encode([
                    [], // First element is empty array
                    [
                        'path' => 'evidence/photos/sample-screenshot.jpg',
                        'name' => 'Security Footage - Key Frame',
                        'uploaded_at' => Carbon::now()->subDays(3)->format('Y-m-d H:i:s'),
                        'uploaded_by' => $user->id,
                    ],
                ]),
                'documents' => json_encode([
                    [], // First element is empty array
                    [
                        'path' => 'evidence/documents/footage-metadata.pdf',
                        'name' => 'Footage Metadata Report',
                        'uploaded_at' => Carbon::now()->subDays(3)->format('Y-m-d H:i:s'),
                        'uploaded_by' => $user->id,
                    ],
                    [
                        'path' => 'evidence/documents/chain-of-custody.pdf',
                        'name' => 'Digital Evidence Chain of Custody Form',
                        'uploaded_at' => Carbon::now()->subDays(3)->format('Y-m-d H:i:s'),
                        'uploaded_by' => $user->id,
                    ],
                ]),
                'chain_of_custody' => json_encode([
                    [
                        'date' => Carbon::now()->subDays(3)->format('Y-m-d H:i:s'),
                        'from_officer_id' => null,
                        'to_officer_id' => $user->id,
                        'action' => 'collected',
                        'notes' => 'Initial collection from security office',
                    ]
                ]),
                'case_id' => $case->id,
                'created_by_user_id' => $user->id,
                'updated_by_user_id' => $user->id,
            ],
            [
                'tag_number' => 'EV-2025-004',
                'evidence_type' => 'documentary',
                'description' => 'Threatening letter',
                'status' => 'in_custody',
                'condition' => 'Paper document with handwritten text',
                'location' => 'Evidence Room A',
                'storage_location' => 'Locker 15, Document Box 3',
                'collection_date' => Carbon::now()->subDays(2),
                'collection_location' => 'Victim\'s mailbox, 123 Main St, Banjul',
                'collection_notes' => 'Found in sealed envelope',
                'collected_by_officer_id' => $user->id,
                'custodian' => $user->id,
                'photos' => json_encode([
                    [], // First element is empty array
                    [
                        'path' => 'evidence/photos/sample-letter.jpg',
                        'name' => 'Threatening Letter - Front',
                        'uploaded_at' => Carbon::now()->subDays(2)->format('Y-m-d H:i:s'),
                        'uploaded_by' => $user->id,
                    ],
                    [
                        'path' => 'evidence/photos/sample-envelope.jpg',
                        'name' => 'Envelope - Front with Postmark',
                        'uploaded_at' => Carbon::now()->subDays(2)->format('Y-m-d H:i:s'),
                        'uploaded_by' => $user->id,
                    ],
                ]),
                'documents' => json_encode([
                    [], // First element is empty array
                    [
                        'path' => 'evidence/documents/handwriting-analysis.pdf',
                        'name' => 'Preliminary Handwriting Analysis',
                        'uploaded_at' => Carbon::now()->subDays(1)->format('Y-m-d H:i:s'),
                        'uploaded_by' => $user->id,
                    ],
                ]),
                'chain_of_custody' => json_encode([
                    [
                        'date' => Carbon::now()->subDays(2)->format('Y-m-d H:i:s'),
                        'from_officer_id' => null,
                        'to_officer_id' => $user->id,
                        'action' => 'collected',
                        'notes' => 'Initial collection from victim\'s mailbox',
                    ]
                ]),
                'case_id' => $case->id,
                'created_by_user_id' => $user->id,
                'updated_by_user_id' => $user->id,
            ],
            [
                'tag_number' => 'EV-2025-005',
                'evidence_type' => 'biological',
                'description' => 'Hair samples',
                'status' => 'in_analysis',
                'condition' => 'Multiple strands in sealed container',
                'location' => 'Forensic Lab',
                'storage_location' => 'Lab 2, Refrigerator 1',
                'collection_date' => Carbon::now()->subDays(4),
                'collection_location' => 'Crime scene bathroom, 123 Main St, Banjul',
                'collection_notes' => 'Found on bathroom floor',
                'collected_by_officer_id' => $user->id,
                'custodian' => $user->id,
                'analyzed_by_officer_id' => $user->id,
                'analysis_date' => Carbon::now()->subDay(),
                'analysis_results' => 'DNA extraction in progress',
                'photos' => json_encode([
                    [], // First element is empty array
                    [
                        'path' => 'evidence/photos/sample-hair.jpg',
                        'name' => 'Hair Samples - Collection Container',
                        'uploaded_at' => Carbon::now()->subDays(4)->format('Y-m-d H:i:s'),
                        'uploaded_by' => $user->id,
                    ],
                    [
                        'path' => 'evidence/photos/sample-hair-microscope.jpg',
                        'name' => 'Hair Sample - Microscope View',
                        'uploaded_at' => Carbon::now()->subDays(1)->format('Y-m-d H:i:s'),
                        'uploaded_by' => $user->id,
                    ],
                ]),
                'documents' => json_encode([
                    [], // First element is empty array
                    [
                        'path' => 'evidence/documents/dna-extraction-protocol.pdf',
                        'name' => 'DNA Extraction Protocol',
                        'uploaded_at' => Carbon::now()->subDays(1)->format('Y-m-d H:i:s'),
                        'uploaded_by' => $user->id,
                    ],
                ]),
                'chain_of_custody' => json_encode([
                    [
                        'date' => Carbon::now()->subDays(4)->format('Y-m-d H:i:s'),
                        'from_officer_id' => null,
                        'to_officer_id' => $user->id,
                        'action' => 'collected',
                        'notes' => 'Initial collection from crime scene',
                    ],
                    [
                        'date' => Carbon::now()->subDays(2)->format('Y-m-d H:i:s'),
                        'from_officer_id' => $user->id,
                        'to_officer_id' => $user->id,
                        'action' => 'transferred',
                        'notes' => 'Transferred to forensic lab for analysis',
                    ]
                ]),
                'case_id' => $case->id,
                'created_by_user_id' => $user->id,
                'updated_by_user_id' => $user->id,
            ],
        ];
        
        // Insert evidence records
        foreach ($evidenceItems as $item) {
            Evidence::firstOrCreate(
                ['tag_number' => $item['tag_number']],
                $item
            );
        }
    }
}
