<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use App\Models\EvidenceTransfer;
use App\Models\CaseEvidence;
use App\Models\User;
use Carbon\Carbon;

class EvidenceTransferSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $evidences = CaseEvidence::all();
        $officers = User::where('is_officer', true)->get();
        
        // Only create transfers for some evidences to demonstrate different scenarios
        $evidencesWithTransfers = $evidences->take(3);
        
        foreach ($evidencesWithTransfers as $index => $evidence) {
            // Get the current custodian from the evidence
            $currentCustodian = User::find($evidence->collected_by_officer_id);
            
            // Skip if no custodian found
            if (!$currentCustodian) {
                continue;
            }
            
            // Find a different officer to transfer to
            $newCustodianIndex = ($index + 1) % count($officers->toArray());
            $newCustodian = $officers[$newCustodianIndex];
            
            // Create first transfer (to forensics)
            $transferDate1 = Carbon::parse($evidence->collection_date)->addDays(1)->format('Y-m-d');
            EvidenceTransfer::create([
                'evidence_id' => $evidence->id,
                'transfer_date' => $transferDate1,
                'transfer_time' => '09:30:00',
                'from_person' => $currentCustodian->name . ' (Badge #' . $currentCustodian->badge_number . ')',
                'to_person' => $newCustodian->name . ' (Badge #' . $newCustodian->badge_number . ')',
                'reason' => 'analysis',
                'notes' => 'Transferred to forensic department for analysis and processing.',
                'recorded_by_user_id' => $currentCustodian->id,
            ]);
            
            // Update chain of custody in the evidence record
            $chainOfCustody = json_decode($evidence->chain_of_custody ?? '[]', true) ?: [];
            $chainOfCustody[] = [
                'date' => $transferDate1,
                'time' => '09:30:00',
                'action' => 'transferred',
                'handler' => $newCustodian->name,
                'badge_number' => $newCustodian->badge_number,
                'location' => 'Forensic Lab',
                'notes' => 'Transferred to forensic department for analysis',
            ];
            $evidence->chain_of_custody = json_encode($chainOfCustody);
            $evidence->save();
            
            // For the first two evidences, create a second transfer (back to evidence room)
            if ($index < 2) {
                // Find another officer to transfer back to
                $returnCustodianIndex = ($index + 2) % count($officers->toArray());
                $returnCustodian = $officers[$returnCustodianIndex];
                
                $transferDate2 = Carbon::parse($transferDate1)->addDays(3)->format('Y-m-d');
                EvidenceTransfer::create([
                    'evidence_id' => $evidence->id,
                    'transfer_date' => $transferDate2,
                    'transfer_time' => '14:15:00',
                    'from_person' => $newCustodian->name . ' (Badge #' . $newCustodian->badge_number . ')',
                    'to_person' => $returnCustodian->name . ' (Badge #' . $returnCustodian->badge_number . ')',
                    'reason' => 'storage',
                    'notes' => 'Analysis complete. Returned to evidence storage.',
                    'recorded_by_user_id' => $newCustodian->id,
                ]);
                
                // Update chain of custody again
                $chainOfCustody[] = [
                    'date' => $transferDate2,
                    'time' => '14:15:00',
                    'action' => 'transferred',
                    'handler' => $returnCustodian->name,
                    'badge_number' => $returnCustodian->badge_number,
                    'location' => 'Evidence Room',
                    'notes' => 'Analysis complete. Returned to evidence storage',
                ];
                $evidence->chain_of_custody = json_encode($chainOfCustody);
                $evidence->save();
            }
            
            // For the first evidence, create a third transfer (to court)
            if ($index === 0) {
                // Transfer to court for presentation
                $courtOfficerIndex = ($index + 3) % count($officers->toArray());
                $courtOfficer = $officers[$courtOfficerIndex];
                
                $transferDate3 = Carbon::parse($transferDate2)->addDays(7)->format('Y-m-d');
                EvidenceTransfer::create([
                    'evidence_id' => $evidence->id,
                    'transfer_date' => $transferDate3,
                    'transfer_time' => '08:45:00',
                    'from_person' => $returnCustodian->name . ' (Badge #' . $returnCustodian->badge_number . ')',
                    'to_person' => $courtOfficer->name . ' (Badge #' . $courtOfficer->badge_number . ')',
                    'reason' => 'court',
                    'notes' => 'Transferred to court for case hearing.',
                    'recorded_by_user_id' => $returnCustodian->id,
                ]);
                
                // Update chain of custody again
                $chainOfCustody[] = [
                    'date' => $transferDate3,
                    'time' => '08:45:00',
                    'action' => 'transferred',
                    'handler' => $courtOfficer->name,
                    'badge_number' => $courtOfficer->badge_number,
                    'location' => 'Banjul High Court',
                    'notes' => 'Transferred to court for case hearing',
                ];
                $evidence->chain_of_custody = json_encode($chainOfCustody);
                $evidence->save();
            }
        }
    }
}
