@extends('adminlte::page')

@section('title', 'Case Report: ' . $case->case_number)

@section('content_header')
    <div class="d-flex justify-content-between">
        <h1>Case Report: {{ $case->case_number }}</h1>
        <div>
            @if($reportType == 'detailed')
                <span class="badge badge-primary">Detailed Report</span>
            @else
                <span class="badge badge-info">Summary Report</span>
            @endif
            <a href="{{ route('case-reports.index', ['case_id' => $case->id]) }}" class="btn btn-secondary ml-2">
                <i class="fas fa-arrow-left"></i> Back to Report Options
            </a>
            <a href="#" onclick="window.print()" class="btn btn-success ml-2">
                <i class="fas fa-print"></i> Print Report
            </a>
        </div>
    </div>
@stop

@section('content')
<div class="row">
    <div class="col-md-12">
        <!-- Report Header -->
        <div class="card">
            <div class="card-header bg-primary">
                <h3 class="card-title">
                    <i class="fas fa-file-alt mr-1"></i> Gambia Police Crime Case Management System
                </h3>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-2 text-center">
                        <img src="{{ asset('img/logo.png') }}" alt="Police Logo" class="img-fluid" style="max-height: 100px;">
                    </div>
                    <div class="col-md-8 text-center">
                        <h2>OFFICIAL CASE REPORT</h2>
                        <h4>{{ $case->station->name ?? 'Unknown Station' }}</h4>
                        <h5>Report Generated: {{ now()->format('F d, Y H:i') }}</h5>
                    </div>
                    <div class="col-md-2 text-center">
                        <div class="bg-light p-2 rounded">
                            <strong>Case #:</strong><br>
                            {{ $case->case_number }}
                        </div>
                    </div>
                </div>
            </div>
        </div>

        @if(in_array('case_details', $includeSections))
        <!-- Case Details -->
        <div class="card">
            <div class="card-header">
                <h3 class="card-title">
                    <i class="fas fa-folder-open mr-1"></i> Case Details
                </h3>
                <div class="card-tools">
                    <span class="badge {{ $case->getStatusBadgeClassAttribute() }}">{{ $case->getStatusLabelAttribute() }}</span>
                    <span class="badge {{ $case->getPriorityBadgeClassAttribute() }}">{{ $case->getPriorityLabelAttribute() }} Priority</span>
                </div>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <table class="table table-bordered">
                            <tr>
                                <th style="width: 30%">Case Title</th>
                                <td>{{ $case->title }}</td>
                            </tr>
                            <tr>
                                <th>Crime Type</th>
                                <td>{{ $crimeTypes[$case->crime_type] ?? $case->crime_type }}</td>
                            </tr>
                            <tr>
                                <th>Incident Date</th>
                                <td>{{ $case->incident_date ? $case->incident_date->format('F d, Y') : 'N/A' }}</td>
                            </tr>
                            <tr>
                                <th>Incident Location</th>
                                <td>{{ $case->incident_location ?? 'N/A' }}</td>
                            </tr>
                            <tr>
                                <th>Reported Date</th>
                                <td>{{ $case->reported_date ? $case->reported_date->format('F d, Y') : 'N/A' }}</td>
                            </tr>
                        </table>
                    </div>
                    <div class="col-md-6">
                        <table class="table table-bordered">
                            <tr>
                                <th style="width: 30%">Police Station</th>
                                <td>{{ $case->station->name ?? 'N/A' }}</td>
                            </tr>
                            <tr>
                                <th>Reporting Officer</th>
                                <td>{{ $case->reportingOfficer->name ?? 'N/A' }}</td>
                            </tr>
                            <tr>
                                <th>Lead Investigator</th>
                                <td>{{ $case->leadInvestigator->name ?? 'N/A' }}</td>
                            </tr>
                            <tr>
                                <th>Case Status</th>
                                <td>{{ $case->getStatusLabelAttribute() }}</td>
                            </tr>
                            <tr>
                                <th>Priority</th>
                                <td>{{ $case->getPriorityLabelAttribute() }}</td>
                            </tr>
                        </table>
                    </div>
                </div>

                <div class="row mt-3">
                    <div class="col-md-12">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="card-title">Case Description</h5>
                            </div>
                            <div class="card-body">
                                <p>{!! nl2br(e($case->description)) !!}</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        @endif

        @if(in_array('victims', $includeSections) && $case->victims->count() > 0)
        <!-- Victims -->
        <div class="card">
            <div class="card-header">
                <h3 class="card-title">
                    <i class="fas fa-user-injured mr-1"></i> Victims ({{ $case->victims->count() }})
                </h3>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-bordered table-striped">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Name</th>
                                <th>Type</th>
                                <th>Contact</th>
                                <th>Injury Status</th>
                                @if($reportType == 'detailed')
                                <th>Address</th>
                                @endif
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($case->victims as $index => $victim)
                            <tr>
                                <td>{{ $index + 1 }}</td>
                                <td>
                                    @if($victim->victim_type == 'individual')
                                        {{ $victim->first_name }} {{ $victim->last_name }}
                                    @else
                                        {{ $victim->organization_name }}
                                    @endif
                                </td>
                                <td>{{ ucfirst($victim->victim_type) }}</td>
                                <td>{{ $victim->contact_phone ?? 'N/A' }}</td>
                                <td>
                                    @if($victim->injury_status)
                                        <span class="badge 
                                            @if($victim->injury_status == 'none') badge-success
                                            @elseif($victim->injury_status == 'minor') badge-info
                                            @elseif($victim->injury_status == 'moderate') badge-warning
                                            @elseif($victim->injury_status == 'severe') badge-danger
                                            @else badge-secondary
                                            @endif">
                                            {{ ucfirst($victim->injury_status) }}
                                        </span>
                                    @else
                                        N/A
                                    @endif
                                </td>
                                @if($reportType == 'detailed')
                                <td>{{ $victim->address ?? 'N/A' }}</td>
                                @endif
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        @endif

        @if(in_array('suspects', $includeSections) && $case->suspects->count() > 0)
        <!-- Suspects -->
        <div class="card">
            <div class="card-header">
                <h3 class="card-title">
                    <i class="fas fa-user-secret mr-1"></i> Suspects ({{ $case->suspects->count() }})
                </h3>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-bordered table-striped">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Name</th>
                                <th>Status</th>
                                @if($reportType == 'detailed')
                                <th>Contact</th>
                                <th>Address</th>
                                <th>Description</th>
                                @endif
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($case->suspects as $index => $suspect)
                            <tr>
                                <td>{{ $index + 1 }}</td>
                                <td>{{ $suspect->first_name }} {{ $suspect->last_name }}</td>
                                <td>{{ ucfirst($suspect->status ?? 'Unknown') }}</td>
                                @if($reportType == 'detailed')
                                <td>{{ $suspect->contact_phone ?? 'N/A' }}</td>
                                <td>{{ $suspect->address ?? 'N/A' }}</td>
                                <td>{{ Str::limit($suspect->description, 100) }}</td>
                                @endif
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        @endif

        @if(in_array('statements', $includeSections) && $case->statements->count() > 0)
        <!-- Statements -->
        <div class="card">
            <div class="card-header">
                <h3 class="card-title">
                    <i class="fas fa-comment mr-1"></i> Statements ({{ $case->statements->count() }})
                </h3>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-bordered table-striped">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Given By</th>
                                <th>Type</th>
                                <th>Date Recorded</th>
                                <th>Verification Status</th>
                                @if($reportType == 'detailed')
                                <th>Statement Content</th>
                                @endif
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($case->statements as $index => $statement)
                            <tr>
                                <td>{{ $index + 1 }}</td>
                                <td>{{ $statement->given_by_name }}</td>
                                <td>{{ ucfirst($statement->statement_type) }}</td>
                                <td>{{ $statement->recorded_date ? $statement->recorded_date->format('Y-m-d') : 'N/A' }}</td>
                                <td>
                                    @if($statement->is_verified)
                                        <span class="badge badge-success">Verified</span>
                                    @else
                                        <span class="badge badge-warning">Unverified</span>
                                    @endif
                                </td>
                                @if($reportType == 'detailed')
                                <td>
                                    <div style="max-height: 150px; overflow-y: auto;">
                                        {!! nl2br(e($statement->statement_content)) !!}
                                    </div>
                                </td>
                                @endif
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        @endif

        @if(in_array('evidence', $includeSections) && $case->evidence->count() > 0)
        <!-- Evidence -->
        <div class="card">
            <div class="card-header">
                <h3 class="card-title">
                    <i class="fas fa-fingerprint mr-1"></i> Evidence ({{ $case->evidence->count() }})
                </h3>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-bordered table-striped">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Tag Number</th>
                                <th>Type</th>
                                <th>Status</th>
                                <th>Collection Date</th>
                                @if($reportType == 'detailed')
                                <th>Description</th>
                                <th>Current Custodian</th>
                                @endif
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($case->evidence as $index => $evidence)
                            <tr>
                                <td>{{ $index + 1 }}</td>
                                <td>{{ $evidence->tag_number }}</td>
                                <td>{{ ucfirst($evidence->evidence_type) }}</td>
                                <td>
                                    <span class="badge 
                                        @if($evidence->status == 'in_custody') badge-success
                                        @elseif($evidence->status == 'in_lab') badge-info
                                        @elseif($evidence->status == 'in_court') badge-warning
                                        @elseif($evidence->status == 'returned') badge-secondary
                                        @elseif($evidence->status == 'destroyed') badge-danger
                                        @else badge-secondary
                                        @endif">
                                        {{ ucfirst(str_replace('_', ' ', $evidence->status)) }}
                                    </span>
                                </td>
                                <td>{{ $evidence->collection_datetime ? $evidence->collection_datetime->format('Y-m-d') : 'N/A' }}</td>
                                @if($reportType == 'detailed')
                                <td>{{ Str::limit($evidence->description, 100) }}</td>
                                <td>{{ $evidence->currentCustodian->name ?? 'N/A' }}</td>
                                @endif
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        @endif

        @if(in_array('updates', $includeSections) && $case->updates->count() > 0)
        <!-- Case Updates -->
        <div class="card">
            <div class="card-header">
                <h3 class="card-title">
                    <i class="fas fa-history mr-1"></i> Case Updates ({{ $case->updates->count() }})
                </h3>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-bordered table-striped">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Date</th>
                                <th>Type</th>
                                <th>Officer</th>
                                <th>Description</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($case->updates as $index => $update)
                            <tr>
                                <td>{{ $index + 1 }}</td>
                                <td>{{ $update->update_date ? $update->update_date->format('Y-m-d') : $update->created_at->format('Y-m-d') }}</td>
                                <td>{{ ucfirst($update->update_type) }}</td>
                                <td>{{ $update->officer->name ?? 'N/A' }}</td>
                                <td>{{ $update->description }}</td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        @endif

        <!-- Report Footer -->
        <div class="card">
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <p><strong>Report Generated By:</strong> {{ Auth::user()->name }}</p>
                        <p><strong>Date & Time:</strong> {{ now()->format('F d, Y H:i') }}</p>
                    </div>
                    <div class="col-md-6 text-right">
                        <p><strong>Case Status:</strong> {{ $case->getStatusLabelAttribute() }}</p>
                        <p><strong>Report ID:</strong> {{ md5($case->id . now()) }}</p>
                    </div>
                </div>
                <hr>
                <div class="row">
                    <div class="col-md-12 text-center">
                        <p class="mb-0"><small>This is an official document of the Gambia Police Crime Case Management System.</small></p>
                        <p><small>CONFIDENTIAL - For official use only.</small></p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@stop

@section('css')
    <link rel="stylesheet" href="/css/admin_custom.css">
    <style>
        @media print {
            .no-print, .main-header, .main-sidebar, .main-footer, .content-header {
                display: none !important;
            }
            .content-wrapper {
                margin-left: 0 !important;
                padding: 0 !important;
            }
            .card {
                break-inside: avoid;
            }
            body {
                padding: 0 !important;
                margin: 0 !important;
            }
            .table-responsive {
                overflow: visible !important;
            }
        }
    </style>
@stop

@section('js')
    <script>
        $(document).ready(function() {
            // Add any JavaScript needed for the report
        });
    </script>
@stop
