@extends('adminlte::page')

@section('title', 'Case Statements')

@section('content_header')
    <div class="d-flex justify-content-between">
        <h1>Case Statements</h1>
        <div>
            @if(request()->has('case_id') && request()->case_id)
                <a href="{{ route('case-statements.create', ['case_id' => request()->case_id]) }}" class="btn btn-primary">
                    <i class="fas fa-plus"></i> Record New Statement
                </a>
            @else
                <a href="{{ route('case-statements.create') }}" class="btn btn-primary">
                    <i class="fas fa-plus"></i> Record New Statement
                </a>
            @endif
        </div>
    </div>
@stop

@section('content')
    @if (session('success'))
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            {{ session('success') }}
            <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                <span aria-hidden="true">&times;</span>
            </button>
        </div>
    @endif

    @if (session('error'))
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            {{ session('error') }}
            <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                <span aria-hidden="true">&times;</span>
            </button>
        </div>
    @endif

    <div class="card">
        <div class="card-header">
            <h3 class="card-title">Filter Statements</h3>
            <div class="card-tools">
                <button type="button" class="btn btn-tool" data-card-widget="collapse">
                    <i class="fas fa-minus"></i>
                </button>
            </div>
        </div>
        <div class="card-body">
            <form action="{{ route('case-statements.index') }}" method="GET">
                <div class="row">
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="case_id">Case</label>
                            <select class="form-control" id="case_id" name="case_id">
                                <option value="">All Cases</option>
                                @foreach($cases as $case)
                                    <option value="{{ $case->id }}" {{ request('case_id') == $case->id ? 'selected' : '' }}>
                                        #{{ $case->case_number }} - {{ Str::limit($case->title, 30) }}
                                    </option>
                                @endforeach
                            </select>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="statement_type">Statement Type</label>
                            <select class="form-control" id="statement_type" name="statement_type">
                                <option value="">All Types</option>
                                <option value="victim" {{ request('statement_type') == 'victim' ? 'selected' : '' }}>Victim</option>
                                <option value="witness" {{ request('statement_type') == 'witness' ? 'selected' : '' }}>Witness</option>
                                <option value="suspect" {{ request('statement_type') == 'suspect' ? 'selected' : '' }}>Suspect</option>
                                <option value="expert" {{ request('statement_type') == 'expert' ? 'selected' : '' }}>Expert</option>
                                <option value="officer" {{ request('statement_type') == 'officer' ? 'selected' : '' }}>Officer</option>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="recorded_by_officer_id">Recorded By</label>
                            <select class="form-control" id="recorded_by_officer_id" name="recorded_by_officer_id">
                                <option value="">All Officers</option>
                                @foreach($officers as $officer)
                                    <option value="{{ $officer->id }}" {{ request('recorded_by_officer_id') == $officer->id ? 'selected' : '' }}>
                                        {{ $officer->name }}
                                    </option>
                                @endforeach
                            </select>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="is_verified">Verification Status</label>
                            <select class="form-control" id="is_verified" name="is_verified">
                                <option value="">All</option>
                                <option value="1" {{ request('is_verified') == '1' ? 'selected' : '' }}>Verified</option>
                                <option value="0" {{ request('is_verified') == '0' ? 'selected' : '' }}>Not Verified</option>
                            </select>
                        </div>
                    </div>
                </div>
                <div class="row">
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="start_date">From Date</label>
                            <input type="date" class="form-control" id="start_date" name="start_date" value="{{ request('start_date') }}">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="end_date">To Date</label>
                            <input type="date" class="form-control" id="end_date" name="end_date" value="{{ request('end_date') }}">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="search">Search</label>
                            <input type="text" class="form-control" id="search" name="search" value="{{ request('search') }}" placeholder="Name, ID, Content...">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group" style="margin-top: 32px;">
                            <button type="submit" class="btn btn-primary float-right">
                                <i class="fas fa-search"></i> Filter
                            </button>
                            <a href="{{ route('case-statements.index') }}" class="btn btn-default float-right mr-2">
                                <i class="fas fa-redo"></i> Reset
                            </a>
                        </div>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <div class="card">
        <div class="card-body">
            <table class="table table-bordered table-striped" id="statements-table">
                <thead>
                    <tr>
                        <th>Case</th>
                        <th>Type</th>
                        <th>Given By</th>
                        <th>Date</th>
                        <th>Recorded By</th>
                        <th>Verified</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    @forelse ($statements as $statement)
                        <tr>
                            <td>
                                <a href="{{ route('cases.show', $statement->case_id) }}">
                                    #{{ $statement->case->case_number }}
                                </a>
                            </td>
                            <td>
                                @switch($statement->statement_type)
                                    @case('victim')
                                        <span class="badge badge-danger">Victim</span>
                                        @break
                                    @case('witness')
                                        <span class="badge badge-info">Witness</span>
                                        @break
                                    @case('suspect')
                                        <span class="badge badge-warning">Suspect</span>
                                        @break
                                    @case('expert')
                                        <span class="badge badge-primary">Expert</span>
                                        @break
                                    @case('officer')
                                        <span class="badge badge-success">Officer</span>
                                        @break
                                    @default
                                        <span class="badge badge-secondary">{{ ucfirst($statement->statement_type) }}</span>
                                @endswitch
                            </td>
                            <td>{{ $statement->given_by_name }}</td>
                            <td>{{ $statement->statement_date->format('d M Y') }}</td>
                            <td>{{ $statement->recordedByOfficer->name }}</td>
                            <td>
                                @if($statement->is_verified)
                                    <span class="badge badge-success">Verified</span>
                                @else
                                    <span class="badge badge-secondary">Not Verified</span>
                                @endif
                            </td>
                            <td>
                                <div class="btn-group">
                                    <a href="{{ route('case-statements.show', $statement->id) }}" class="btn btn-sm btn-info">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                    <a href="{{ route('case-statements.edit', $statement->id) }}" class="btn btn-sm btn-warning">
                                        <i class="fas fa-edit"></i>
                                    </a>
                                    @if(!$statement->is_verified && (Auth::user()->is_admin || Auth::id() != $statement->recorded_by_officer_id))
                                        <a href="{{ route('case-statements.verify', $statement->id) }}" class="btn btn-sm btn-success">
                                            <i class="fas fa-check"></i>
                                        </a>
                                    @endif
                                    <button type="button" class="btn btn-sm btn-danger" data-toggle="modal" data-target="#deleteModal{{ $statement->id }}">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>

                                <!-- Delete Modal -->
                                <div class="modal fade" id="deleteModal{{ $statement->id }}" tabindex="-1" role="dialog" aria-labelledby="deleteModalLabel" aria-hidden="true">
                                    <div class="modal-dialog" role="document">
                                        <div class="modal-content">
                                            <div class="modal-header">
                                                <h5 class="modal-title" id="deleteModalLabel">Delete Statement</h5>
                                                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                                                    <span aria-hidden="true">&times;</span>
                                                </button>
                                            </div>
                                            <div class="modal-body">
                                                Are you sure you want to delete this statement from <strong>{{ $statement->given_by_name }}</strong>?
                                                <p class="text-danger mt-2">This action cannot be undone and will permanently delete all associated files.</p>
                                            </div>
                                            <div class="modal-footer">
                                                <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                                                <form action="{{ route('case-statements.destroy', $statement->id) }}" method="POST" style="display: inline-block;">
                                                    @csrf
                                                    @method('DELETE')
                                                    <button type="submit" class="btn btn-danger">Delete Statement</button>
                                                </form>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="7" class="text-center">No statements found.</td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
    </div>

    <div class="d-flex justify-content-center">
        {{ $statements->appends(request()->query())->links() }}
    </div>
@stop

@section('css')
    <link rel="stylesheet" href="/css/admin_custom.css">
@stop

@section('js')
    <script>
        $(document).ready(function() {
            $('#statements-table').DataTable({
                "paging": false,
                "lengthChange": false,
                "searching": false,
                "ordering": true,
                "info": false,
                "autoWidth": false,
                "responsive": true,
            });
        });
    </script>
@stop
