@extends('adminlte::page')

@section('title', 'Backup Settings')

@section('content_header')
    <div class="d-flex justify-content-between">
        <h1><i class="fas fa-database mr-2"></i>Database Backup & Restore</h1>
    </div>
@stop

@section('content')
<div class="row">
    <div class="col-md-12">
        <div class="card card-primary card-outline">
            <div class="card-header">
                <h3 class="card-title"><i class="fas fa-plus-circle mr-1"></i> Create New Backup</h3>
                <div class="card-tools">
                    <button type="button" class="btn btn-tool" data-card-widget="collapse" title="Collapse">
                        <i class="fas fa-minus"></i>
                    </button>
                </div>
            </div>
            
            @if(session('success'))
                <div class="alert alert-success alert-dismissible">
                    <button type="button" class="close" data-dismiss="alert" aria-hidden="true">&times;</button>
                    <h5><i class="icon fas fa-check"></i> Success!</h5>
                    {{ session('success') }}
                </div>
            @endif
            
            @if(session('error'))
                <div class="alert alert-danger alert-dismissible">
                    <button type="button" class="close" data-dismiss="alert" aria-hidden="true">&times;</button>
                    <h5><i class="icon fas fa-ban"></i> Error!</h5>
                    {{ session('error') }}
                </div>
            @endif
            
            <div class="card-body">
                <div class="row">
                    <div class="col-md-12">
                        <p>Create a backup of the entire database. This will help you restore the system in case of data loss.</p>
                        <form action="{{ route('settings.create-backup') }}" method="POST">
                            @csrf
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-download mr-1"></i> Create Database Backup
                            </button>
                        </form>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="card card-info card-outline">
            <div class="card-header">
                <h3 class="card-title"><i class="fas fa-list mr-1"></i> Available Backups</h3>
                <div class="card-tools">
                    <button type="button" class="btn btn-tool" data-card-widget="collapse" title="Collapse">
                        <i class="fas fa-minus"></i>
                    </button>
                </div>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-bordered table-striped">
                        <thead>
                            <tr>
                                <th>Backup File</th>
                                <th>Size</th>
                                <th>Created At</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @forelse($backups as $backup)
                                <tr>
                                    <td>{{ $backup['name'] }}</td>
                                    <td>{{ round($backup['size'] / 1024, 2) }} KB</td>
                                    <td>{{ date('Y-m-d H:i:s', $backup['date']) }}</td>
                                    <td>
                                        <div class="btn-group">
                                            <a href="{{ route('settings.download-backup', $backup['name']) }}" class="btn btn-sm btn-info" data-toggle="tooltip" title="Download this backup file">
                                                <i class="fas fa-download"></i>
                                            </a>
                                            <button type="button" class="btn btn-sm btn-warning" data-toggle="modal" data-target="#restoreModal{{ str_replace('.', '_', $backup['name']) }}" title="Restore from this backup">
                                                <i class="fas fa-sync-alt"></i>
                                            </button>
                                            <form action="{{ route('settings.delete-backup', $backup['name']) }}" method="POST" class="d-inline">
                                                @csrf
                                                @method('DELETE')
                                                <button type="submit" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to delete this backup?')" data-toggle="tooltip" title="Delete this backup">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </form>
                                        </div>
                                        
                                        <!-- Restore Modal -->
                                        <div class="modal fade" id="restoreModal{{ str_replace('.', '_', $backup['name']) }}">
                                            <div class="modal-dialog">
                                                <div class="modal-content">
                                                    <div class="modal-header bg-warning">
                                                        <h4 class="modal-title"><i class="fas fa-sync-alt mr-2"></i> Restore Database</h4>
                                                        <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                                                            <span aria-hidden="true">&times;</span>
                                                        </button>
                                                    </div>
                                                    <div class="modal-body">
                                                        <p>Are you sure you want to restore the database from backup file <strong>{{ $backup['name'] }}</strong>?</p>
                                                        <p><strong>Warning:</strong> This will overwrite all current data with the data from this backup. This action cannot be undone.</p>
                                                    </div>
                                                    <div class="modal-footer justify-content-between">
                                                        <button type="button" class="btn btn-default" data-dismiss="modal"><i class="fas fa-times mr-1"></i> Cancel</button>
                                                        <a href="{{ route('settings.restore-backup', $backup['name']) }}" class="btn btn-warning"><i class="fas fa-sync-alt mr-1"></i> Restore</a>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <!-- Delete Modal -->
                                        <div class="modal fade" id="deleteModal{{ str_replace('.', '_', $backup['name']) }}">
                                            <div class="modal-dialog">
                                                <div class="modal-content">
                                                    <div class="modal-header bg-danger">
                                                        <h4 class="modal-title">Confirm Delete</h4>
                                                        <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                                                            <span aria-hidden="true">&times;</span>
                                                        </button>
                                                    </div>
                                                    <div class="modal-body">
                                                        <p>Are you sure you want to delete this backup?</p>
                                                        <p>This action cannot be undone.</p>
                                                    </div>
                                                    <div class="modal-footer justify-content-between">
                                                        <button type="button" class="btn btn-default" data-dismiss="modal"><i class="fas fa-times mr-1"></i> Cancel</button>
                                                        <form action="{{ route('settings.delete-backup', $backup['name']) }}" method="POST" style="display: inline;">
                                                            @csrf
                                                            @method('DELETE')
                                                            <button type="submit" class="btn btn-danger"><i class="fas fa-trash mr-1"></i> Delete</button>
                                                        </form>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </td>
                                </tr>
                            @empty
                                <tr>
                                    <td colspan="4" class="text-center">No backups available</td>
                                </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        
        <div class="card card-secondary card-outline">
            <div class="card-header">
                <h3 class="card-title"><i class="fas fa-calendar-alt mr-1"></i> Backup Schedule</h3>
                <div class="card-tools">
                    <button type="button" class="btn btn-tool" data-card-widget="collapse" title="Collapse">
                        <i class="fas fa-minus"></i>
                    </button>
                </div>
            </div>
            <div class="card-body">
                <form action="{{ route('settings.update-backup-settings') }}" method="POST">
                    @csrf
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="auto_backup_enabled">Automatic Backups</label>
                                <div class="input-group">
                                    <div class="input-group-prepend">
                                        <span class="input-group-text"><i class="fas fa-clock"></i></span>
                                    </div>
                                    <select class="form-control select2" id="auto_backup_enabled" name="auto_backup_enabled">
                                        <option value="1" {{ \App\Models\Setting::getValue('auto_backup_enabled', '0') == '1' ? 'selected' : '' }}>Enabled</option>
                                        <option value="0" {{ \App\Models\Setting::getValue('auto_backup_enabled', '0') == '0' ? 'selected' : '' }}>Disabled</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="backup_frequency">Backup Frequency</label>
                                <div class="input-group">
                                    <div class="input-group-prepend">
                                        <span class="input-group-text"><i class="fas fa-calendar"></i></span>
                                    </div>
                                    <select class="form-control select2" id="backup_frequency" name="backup_frequency">
                                        <option value="daily" {{ \App\Models\Setting::getValue('backup_frequency', 'weekly') == 'daily' ? 'selected' : '' }}>Daily</option>
                                        <option value="weekly" {{ \App\Models\Setting::getValue('backup_frequency', 'weekly') == 'weekly' ? 'selected' : '' }}>Weekly</option>
                                        <option value="monthly" {{ \App\Models\Setting::getValue('backup_frequency', 'weekly') == 'monthly' ? 'selected' : '' }}>Monthly</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="backup_retention_days">Retention Period (days)</label>
                                <div class="input-group">
                                    <div class="input-group-prepend">
                                        <span class="input-group-text"><i class="fas fa-trash-alt"></i></span>
                                    </div>
                                    <input type="number" class="form-control" id="backup_retention_days" name="backup_retention_days" value="{{ \App\Models\Setting::getValue('backup_retention_days', '30') }}" min="1" max="365" placeholder="Enter days to retain backups">
                                </div>
                                <small class="form-text text-muted"><i class="fas fa-info-circle mr-1"></i> Backups older than this will be automatically deleted</small>
                            </div>
                        </div>
                    </div>
                    
                    <button type="submit" class="btn btn-primary" data-toggle="tooltip" title="Save backup schedule settings"><i class="fas fa-save mr-1"></i> Save Schedule Settings</button>
                    <a href="{{ route('settings.index') }}" class="btn btn-default" data-toggle="tooltip" title="Return to settings menu"><i class="fas fa-times mr-1"></i> Cancel</a>
                </form>
            </div>
        </div>
    </div>
</div>
@stop

@section('css')
    <link rel="stylesheet" href="/css/admin_custom.css">
@stop

@section('js')
    <script>
        $(document).ready(function() {
            // Initialize select2
            $('.select2').select2({
                theme: 'bootstrap4'
            });
            
            // Initialize tooltips
            $('[data-toggle="tooltip"]').tooltip();
            
            // Add fade-out effect to alerts after 5 seconds
            setTimeout(function() {
                $('.alert').fadeOut('slow');
            }, 5000);
        });
    </script>
@stop
